classdef fge_test < meq_test
  % Tests for basic fge behaviour
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

  properties (TestParameter)
    debuglevel = {'partial','full'}
  end

  methods(Test,TestTags = {'fge'},ParameterCombination='sequential')

    function test_fget_iterator(testCase)
      % test LY equality between regular fget call and call as part of an iterator

      add_delay = false;
      PP = {'lin',false};

      % Call default vs iterator fget modes and return both results
      [L,LX,LY,LYo] = testCase.run_default_vs_iterator(PP,add_delay);

      % Check LY,LYo equivalence
      testCase.verifyTrue(structcmp(LY,LYo,sqrt(eps),{'shot','tokamak'}),...
        'Differences in LY structures higher than expected tolerance');
      testCase.verifyEqual(LY.Va,LX.Va,'expected direct passthrough without delays')
    end

    function test_fge_delays(testCase)
      add_delay = true;
      PP = {};
      [L,LX,LY,LYo] = testCase.run_default_vs_iterator(PP,add_delay);

      testCase.verifyTrue(structcmp(LY,LYo,sqrt(eps),{'shot','tokamak'}),...
        'Differences in LY structures higher than expected tolerance');

      for ia = 1:L.G.na
        dt = mean(diff(LX.t));
        idelay = round(L.G.Vadelay(ia)/dt);
        testCase.assertEqual(LX.Va(ia,1:end-idelay),LY.Va(ia,(1+idelay):end),...
          sprintf('output Va delay on channel %d is not as expected',ia));
        if idelay>0
          % first entry equal to initialized value from LX
          testCase.verifyTrue(all(LY.Va(ia,1:idelay)==LX.Va(ia,1)),...
            'unexpected value for LY.Va before power supply delay')
        end
      end
    end

    function test_fge_nonconvergence_recovery(testCase)
      PP = {'nnoc',1,'tolF',1e-12,'nrestart',1,'kmax',1,'keepbest',true,...
        'debug',1};
      add_delay = false;
      [L,~,LY,LYo] = testCase.run_default_vs_iterator(PP,add_delay);
      testCase.assertTrue(structcmp(LY,LYo,sqrt(eps),{'shot','tokamak'}),...
        'Differences in LY structures higher than expected tolerance');

      testCase.assertEqual(numel(LY.t),2+L.P.nnoc,'expected 3 time slices')
      testCase.assertFalse(any(LY.isconverged(2:end)),'expected non-convergence')
    end

    function test_debug_and_plotting(testCase,debuglevel)

      % call with various debugging levels to test display / plotting
      [L,LX] = fge('ana',1,[0,1e-4]);

      if strcmp(debuglevel,'partial')
        L.P.debug = 1;
        L.P.debugplot = 1;
      elseif strcmp(debuglevel,'full')
        L.P.debug = intmax;
        L.P.debugplot = intmax;
      end
      testCase.assertWarningFree(@()fget(L,LX));
    end

    function test_multiple_initial_guesses(testCase)
      [L,LX] = fge('ana',1,[0,1e-4],...
        'initguess',{'previous','extrap','vacuum'});
      L.P.debug = 3;
      LY = fget(L,LX);
      testCase.assertNotEmpty(LY)
    end

    function test_keepok(testCase)
      % test for keeping good enough solution following non-convergence
      [L,LX] = fge('ana',1,[0,1e-4],... % two time slices means one slice after initial condition
        'kmax',1,... % only 1 Newton iteration so we expect non-convergence
        'nnoc',0); % restrict iterations and allowed non-converged steps
      LX.Va = rand(size(LX.Va));

      L.P.debug = 2;
      LY = fget(L,LX);
 
      L.P.keepok = true;
      LY2 = fget(L,LX);

      testCase.verifyFalse( LY.isconverged(2),'expected non-converged second time slice')
      testCase.verifyFalse(LY2.isconverged(2),'expected non-converged second time slice')

      testCase.verifyGreaterThan(LY2.resC(2),LY.resC(2),'expected smaller residual with keepok=true')
      testCase.verifyGreaterThan(LY2.resy(2),LY.resy(2),'expected smaller residual with keepok=true')
      testCase.verifyGreaterThan(LY2.rese(2),LY.rese(2),'expected smaller residual with keepok=true')
      testCase.verifyGreaterThan(LY2.resFx(2),LY.resFx(2),'expected smaller residual with keepok=true')
    end

    function test_assign_Iy(testCase)
      % Test direct assignment of Iy
      tt = 0.01:10e-3:0.1; nt = numel(tt);
      [L,LX] = fge('ana',1,tt);
      LX.t = tt;
      LX0 = meqxk(LX,1); % one slice

      LX.Ip = linspace(0.9,1,nt).*LX0.Ip;
      LX.Iy = LX0.Iy.*reshape((LX.Ip./LX0.Ip),1,1,nt);

      L.P.Ipmin = LX.Ip(end);

      % Check that Iy are as prescribed when not solving for it
      LY = fget(L,LX);

      ieq = LX.Ip<L.P.Ipmin; % expected equal indices
      testCase.assertEqual(LY.Iy(:,:,ieq),LX.Iy(:,:,ieq),...
        'AbsTol',L.P.tolF*L.Iy0,...
        'expected equal Iy when assigning')

      testCase.assertGreaterThan(norm(LY.Iy(:,:,~ieq)-LX.Iy(:,:,~ieq)),...
        L.Iy0*L.P.tolF,'expected different Iy when solving');
    end
  end

  methods (Static)
    function [L,LX,LY,LYo] = run_default_vs_iterator(PP,add_delay)
      dt = 0.5e-3;
      nt = 11;
      t = dt*(0:nt-1); % time
      [L,LX] = fge('ana',2,t,PP{:});

      LX.Va = 100*randn(L.G.na,nt);
      
      if add_delay
        L.G.Vadelay(:) = dt*(0:(L.G.na-1));
      end
      % run in default mode
      LYo = fget(L,LX);

      % run in iterator mode
      % init
      [LYt,xnl,xnldot,Prec,psstate,Tstate,nnoc] = fget(L,meqxk(LX,1),'dt',dt);
      LY(1) = LYt; % init
      % iterate
      for ii=2:numel(LX.t)
        LXt = meqxk(LX,ii);
        % run as iterator
        [LYt,xnl,xnldot,Prec,psstate,Tstate,nnoc] = fget(L,LXt,...
          'LYp',LYt,...
          'xnl',xnl,...
          'xnldot',xnldot,...
          'Prec',Prec,...
          'dt',dt,...
          'psstate',psstate,...
          'Tstate',Tstate,...
          'nnoc',nnoc);
        LY(ii) = LYt; %#ok<AGROW>
        if nnoc>L.P.nnoc; break; end
      end
      LY = meqlpack(LY);
    end
  end
end