classdef fge_jacobian_comp_test < fgs_fge_jacobian_comp_test
  % Compares FGE analytical Jacobian with finite-difference Jacobian
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

  properties(TestParameter)
    algosNL = {'all-nl', 'all-nl-Fx','Newton-GS'};
    shot = struct('diverted',2,'double_null',4,'squashed',5);
    % Restrict combination of agcon for FGE to reduce total number of cases
    agcon = struct('Ip_bp_li',{{'Ip','bp','li'}},...  % Need to nest cell arrays to avoid creating structure arrays
                   'Ip_bp_qA',{{'Ip','bp','qA'}});
    cdetype = {'','cde_ss_0D','OhmTor_0D','OhmTor_rigid_0D','OhmTor_rigid_0D_noLpext','cde_1D','cde_OhmTor_1D','static_cde_1D','static_Iy_1D'};
    % Separate multiple domain cases as they typically run with fewer bf
    shot_multiD = struct(meq_jacobian_test.doublets_under_test{:});
    %
    % Disable fgeF convergence test for now
    convergence_test = struct('false',false);
  end

  methods
    function setup_code(testCase, algosNL)
      testCase.code = 'fge';
      testCase.algoNL = algosNL;
      % State perturbation amplitude (using already existing deltaFx)
      if strcmpi(algosNL,'all-nl')
        testCase.deltaFx = 1e-5;
        testCase.meshDeltaFx = logspace(-5,-4,3);
        % Even though this works for some cases, it seems like the error
        % falls of a cliff around dx~1e-4. This does not happen for FGS
        % but is present in Iy_jacobian_test so this should definitely be
        % investigated.
      else % all-nl-Fx, Newton-GS
        testCase.deltaFx = 1e-6;
        testCase.meshDeltaFx = 0.5*logspace(-6,-5,3);
      end
    end
  end

  methods(Test,TestTags={'Jacobian-Integration'},ParameterCombination='pairwise')
    function test_code_jacobian_vs_fd_singleD(testCase,algosNL,shot,agcon,cdetype,convergence_test)

      % Run setup
      testCase.setup_code(algosNL);
      
      % Adjust iLpext
      iLpext = true;
      if contains(cdetype,'_noLpext')
        cdetype = erase(cdetype,'_noLpext');
        iLpext  = false;
      end

      % remove Ip constraint if cde
      if endsWith(cdetype, '0D')
        agcon = agcon(~strcmp('Ip', agcon));
      elseif endsWith(cdetype, '1D')
        % 1D cde needs only bp constraint
        agcon = {{'bp'}};
      end

      PP = {'agcon',agcon,'cde',cdetype,'iLpext',iLpext,'ssinit',false,'ilu_droptol',1e-4};

      % Run initial setup
      [L,LX] = get_L_LX(testCase,shot,testCase.t,PP{:});

      % Call generic routine
      testCase.test_code_jacobian_vs_fd(L,LX,convergence_test);
    end

    function test_code_jacobian_vs_fd_multiD(testCase,algosNL,shot_multiD,cdetype,convergence_test)

      % Run setup
      testCase.setup_code(algosNL);

      iLpext = true;
      if contains(cdetype,'_noLpext')
        cdetype = erase(cdetype,'_noLpext');
        iLpext  = false;
      end

      % Set agcon for no cde and cde case
      if isempty(cdetype)
        agcon_multiD = {'Ip', 'bp', 'qA'};
      elseif endsWith(cdetype, '0D')
        agcon_multiD = {'bp', 'qA'};
      elseif endsWith(cdetype, '1D')
        agcon_multiD = {'bp'};
      end

      PP = {'agcon',agcon_multiD,'cde',cdetype,'iLpext',iLpext,'ssinit',false,'ilu_droptol',1e-5};
      
      % Run initial setup
      [L,LX] = get_L_LX(testCase,shot_multiD,testCase.t,PP{:});

      % Call generic routine
      testCase.test_code_jacobian_vs_fd(L,LX,convergence_test);
    end
  end

end
