classdef fge_CDE_test < meq_test
  % Tests of FGE with current diffusion equation
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  properties
    verbosity = 0; % Make plot 1 to make the plot

    tok = 'ana';
    shot = 1;
    tstart = 0;
    tend = 1e-2;
    dt = 5e-4;
    nu = 30; %
    wcompa = 0; % no regularization in compensation currents
    
    L, Lss, LX, LXss;
  end
  
  properties (ClassSetupParameter)
   cdetype = {'cde_ss_0D', 'OhmTor_0D', 'OhmTor_rigid_0D', 'cde_OhmTor_1D', 'static_cde_1D', 'static_Iy_1D', 'cde_1D'};
  end
  
  methods (TestClassSetup)
    function setup_LLX(testCase, cdetype)
      t = testCase.tstart:testCase.dt:testCase.tend;
      
      [testCase.L, testCase.LX] = fge(testCase.tok,...
        testCase.shot,t,...
        'ssinit',false,...
        'debug',testCase.verbosity,... % debug ouput level of simulation
        'selu','e','nu',testCase.nu,...
        'izgrid',true,...
        'wcompa',testCase.wcompa);
      
      if endsWith(cdetype, '1D')
        agcon = {'bp'};
      else
        agcon = {'bp', 'qA'};
      end
      
      [testCase.Lss, testCase.LXss]  = fge(testCase.tok,...
        testCase.shot,t,...
        'ssinit',false,...
        'debug',testCase.verbosity,... % debug ouput level of simulation
        'selu','e','nu',testCase.nu,...
        'izgrid',true,...
        'agcon',agcon,...
        'cde',cdetype,...
        'wcompa',testCase.wcompa);
    end
  end
  
  methods (Test, TestTags = {'fge'})
    
    function test_initial_conditions(testCase)
      L = testCase.Lss; LX = meqxk(testCase.LXss,1);

      %% LX with steady-state initial condition
      LX2 = fgex(L.P.tokamak,LX.t,L,LX);

      %% Same but with focus on LY region only for field compensation
      L.P.compregion = 'Iy';
      LX3 = fgex(L.P.tokamak,LX.t,L,LX);
      
      if testCase.verbosity
        clf;
        meqcompare(L,LX,LX2,LX3);
        % inner flux surfaces are not touched, while vessel currents added.
      end
      
      % check that new equilibrium Iy is close to previous one as far as plasma is concerned.
      FAB = LX.FA-LX.FB;
      testCase.verifyLessThan(norm((LX.Iy-LX2.Iy)/LX.Ip),1e-4,'Iy mismatch');
      testCase.verifyLessThan(norm((LX.Iy-LX3.Iy)/LX.Ip),1e-4,'Iy mismatch');
      testCase.verifyLessThan(norm(LX.FA-LX2.FA)/FAB,1e-3,'FA mismatch');
      testCase.verifyLessThan(norm(LX.FA-LX3.FA)/FAB,1e-3,'FA mismatch');
      testCase.verifyLessThan(norm(LX.FB-LX2.FB)/FAB,1e-3,'FB mismatch');
      testCase.verifyLessThan(norm(LX.FB-LX3.FB)/FAB,1e-3,'FB mismatch');
    end
    
    function test_steady_state(testCase)
      testCase.assumeTrue(endsWith(testCase.Lss.P.cde, '0D'), '1D cde steady state init not yet supported');
      % Check that steady-state conditions are maintained in time-dependent
      % simulation
      L = testCase.Lss; LX = testCase.LXss;
      L.P.ssinit = true;
      LX = fgex(L.P.tokamak,LX.t,L,LX); % compute new initial condition
      
      nt = numel(LX.t);
      % FGE run
      LY = fget(L,LX);
      LYss = meqlpack(repmat(meqxk(LY,1),1,nt));

      % Check that simulation converged
      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),'FGE simulation did not converge, aborting ...')
      % Check that initial equilibrium is maintained
      tol = 1e-4;
      testCase.verifyEqual(LY.Ip,LYss.Ip,'AbsTol',tol*L.Ip0);
      testCase.verifyEqual(LY.Iu,LYss.Iu,'AbsTol',tol*L.Iu0);
      testCase.verifyEqual(LY.zA,LYss.zA,'AbsTol',tol*L.P.r0);
      testCase.verifyEqual(LY.rA,LYss.rA,'AbsTol',tol*L.P.r0);
      testCase.verifyEqual(LY.FB-LY.FA,LYss.FB-LYss.FA,'AbsTol',tol*L.Fx0);

      % Check that the currents are as expected from initialization
      testCase.verifyEqual(LY.Ia,LX.Ia,'AbsTol',tol*L.Ia0)
    end
    
    function test_fge_CDE_full_vs_ss(testCase)
      % Test open-loop FGE with CDE with various constraints   
   
      testCase.assumeFail('Ignoring test for now until we find a good test for SS vs Full CDE')
      
      LX = testCase.LX; %#ok<*PROP>
      Lss = testCase.Lss;
      L = testCase.L;
      
      L.P.debug = 1;
      
      LX.Ip  = repmat(LX.Ip(1),  1, numel(LX.t));
      LX.rBt = repmat(LX.rBt(1), 1, numel(LX.t));
      LX.bp  = repmat(LX.bp(1),  1, numel(LX.t));
      LX.qA  = repmat(LX.qA(1),  1, numel(LX.t));
      
      LYss = fget(Lss,LX);
      LY   = fget(L,LX);
      
      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),'FGE did not converge');
      testCase.verifyEqual(LY.Ip(end),LYss.Ip(end),'RelTol',1e-2,'Ip(end) tolerance exceeded')
      testCase.verifyEqual(LY.Ip(1),LYss.Ip(1),'Ip(1) tolerance exceeded')
      testCase.verifyEqual(LY.zA,LYss.zA,'RelTol',1e-3,'zA tolerance exceeded')
      testCase.verifyEqual(LY.rA,LYss.rA,'RelTol',1e-3,'rA tolerance exceeded')
      
      if testCase.verbose
        clf;
        meqplotevo(L,LY,LYss,'legstr',{'full','SS'});
        drawnow;
      end
    end
    
  
  end
end
