classdef fbtxinterp_tests < meq_test
  % Tests for fbtxinterp.m
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  properties
    verbosity = 0;
  end
  
  properties(TestParameter)
    field_to_check = {'gpf','gpi','gpb','gpc','gpv','gpu','gpd','gpa'};
    suffix         = {'e','d'};
    interpmethod   = {'none','nearest','previous','next'}; % interpolation method
  end
  
  methods(Test,TestTags={'fbt'})
    function test_interpolation_between_equal_slices(testCase)
      [~,LX] = fbt('ana',1,[0,0]);
      LX.t = [0,1];
      ti = 0.5; % interpolation time
      cpts = true; % interpolate control points
      LXi = fbtxinterp(LX,ti,'linear',cpts); 
      LX_exp = meqxk(LX,1); LX_exp.t=ti;
      testCase.assertEqual(LX_exp,LXi,'expected equal LX');
    end
    
    function test_interpolation_changing_values(testCase)
      [L,LX] = fbt('ana',1,[0,0]); LX.t = [0 1];
      LX.gpfa(1,end) = 0.02; % offset one flux point
      ti = linspace(0,1,11);
      cpts = true; % interpolate control points too
      LX = fbtxinterp(LX,ti,'linear',cpts);
      % solve whole thing
      LXi = fbtt(L,LX);
      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LXi),'some slices did not converge');
      
      if testCase.verbosity > 1
        for ii=1:numel(LXi.t)
          clf;
          fbtplot(L,meqxk(LX,ii),meqxk(LXi,ii));
          drawnow;
        end
      end
    end
    
    function test_unordered_times(testCase)
      % Check that unordered LX slices gives expected error
      [~,~] = testCase.assertError(@() fbt('ana',1,[1 0]),'fbtxinterp:TimeNotSorted');
      
      % Check that unordered time gives expected error
      [~,LX] = fbt('ana',1,[0 1]); tt = [0.5,0.2];
      testCase.assertError(@() fbtxinterp(LX,tt),'fbtxinterp:TimeNotSorted')
    end
    
    function test_not_allowed_slices(testCase)
      [~,LXi(1)] = fbt('ana',1);
      [~,LXi(2)] = fbt('ana',2);
      LX = meqlpack(LXi); LX.t = [0,1]; % assign time
      % Check that extrapolating outside time range throws expected error
      testCase.assertError(@() fbtxinterp(LX, 2),'fbtxinterp:OutOfTimeRange');
      testCase.assertError(@() fbtxinterp(LX,-1),'fbtxinterp:OutOfTimeRange');
      
      % Check that interpolating incompatible time slices with control points throws expected error
      cpts = true; % attempt to interpolate control points
      testCase.assertError(@() fbtxinterp(LX,0.5,'linear',cpts),'fbtxinterp:NaNMismatch');

      % Check that interpolating incompatible slices without control points works
      cpts = false; % do not attempt to interpolate control points
      LXi = fbtxinterp(LX,0.5,'linear',cpts);
      % expect NaNs for gpr, gpz in this interpolated time point
      testCase.assertTrue(all(isnan(LXi.gpr)) & all(isnan(LXi.gpz)),'gpr,gpz should be NaN')
    end
    
    function test_methods(testCase,interpmethod)
      [~,LX] = fbt('ana',1,[0,0],'selu','e','nu',20); % w. vessel to include gpua
      LX.t = [0,1]; % time grid
      t = linspace(LX.t(1),LX.t(end),11); % new time grid with extrapolation
      switch interpmethod
        case 'none'
          testCase.verifyError(@() fbtxinterp(LX,1,interpmethod),'fbtxinterp:NotAllowed');
        otherwise
          LXi = fbtxinterp(LX,t,interpmethod);
          % check expected time base depending on interpolation method
          switch interpmethod
            case 'linear',   it = t;
            case 'next',     it = iceil(LX.t,t); t_expected = LX.t(it);
            case 'previous', it = ifloor(LX.t,t); t_expected = LX.t(it);
            case 'nearest',  it = iround(LX.t,t); t_expected = LX.t(it);
          end
          testCase.assertEqual(t_expected,LXi.t,'interpolated time base not as expected')
          % check that equilibria are at correct index
          for ii=1:numel(it)
            LX_exp = meqxk(LX,it(ii)); LX_exp.t = t_expected(ii);
            testCase.verifyEqual(LX_exp,meqxk(LXi,ii),'interpolated equilibrium not as expected')
          end
      end
    end
  end
end