classdef fbttimeder_test < meq_test
% Tests of interpolation stencils for computing time derivatives in FBT
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

properties
  t_ = [0,0.02,0.05,0.1,0.2,0.8,0.9,0.92];
end

properties(TestParameter)
  type = {'centered','backward'}
end

methods(Test,TestTags={'Unit'})
  function test_fbttimeder(testCase,type)
    % Check fbttimeder outputs using Taylor expansion formula
    %   A Taylor expansion around t(it) is performed for f(t(it-1)) and
    %   f(t(it+1)) up to 2nd order. They are then combined using the d1dt
    %   and d2dt stencils, and we check that the zero-th order terms are 0
    %   for d1dt and d2dt, the 1st order terms are 1 for d1dt and 0 for
    %   d2dt, and the 2nd order terms are 0 for d1dt (type=centered only)
    %   and 1 for d2dt
    t = testCase.t_;
    
    [d1dt,d2dt] = fbttimeder(t,type);
    
    nt = numel(t);
    dt = diff(t);
    
    % Check interpolation using Taylor expansion around t(it)
    test_d1dt = zeros(3,nt-2);
    test_d2dt = zeros(3,nt-2);
    % Compute coefficients of Taylor expansion 
    for it = 2:nt-1
      % Coefficients for f'
      x1 = zeros(nt,1); x1(it-1) = -dt(it-1); x1(it+1) = dt(it);
      % Coefficients for f''
      x2 = 0.5*x1.^2;
      % Terms for f,f' and f'' using d1dt
      test_d1dt(1,it-1) = sum(d1dt(it,:)   );
      test_d1dt(2,it-1) = sum(d1dt(it,:)*x1);
      test_d1dt(3,it-1) = sum(d1dt(it,:)*x2);
      % Terms for f,f' and f'' using d2dt
      test_d2dt(1,it-1) = sum(d2dt(it,:)   );
      test_d2dt(2,it-1) = sum(d2dt(it,:)*x1);
      test_d2dt(3,it-1) = sum(d2dt(it,:)*x2);
    end
    % Expected coefficients of Taylor expansion
    ref_d1dt = zeros(3,nt-2); ref_d1dt(2,:) = 1;
    ref_d2dt = zeros(3,nt-2); ref_d2dt(3,:) = 1;
    % Backward type doesn't zero out f'' terms
    if strcmp(type,'backward')
      test_d1dt(3,:) = 0;
    end
    % Compare
    msg = 'Interpolation formula for %s and type=%s fails test';
    testCase.verifyEqual(test_d1dt,ref_d1dt,'AbsTol',sqrt(eps),sprintf(msg,'d1dt',type));
    testCase.verifyEqual(test_d2dt,ref_d2dt,'AbsTol',sqrt(eps),sprintf(msg,'d2dt',type));
  end
  
  function test_parabola_fbttimeder(testCase)
    % Check centered fbttimeder outputs for parabolic case
    t = testCase.t_(:);
    
    c = rand(3,1);
    Y = polyval(c,t);
    c1 = polyder(c);
    dYdt = polyval(c1,t);
    c2 = polyder(c1);
    d2Ydt2 = polyval(c2,t);
    
    [d1dt,d2dt] = fbttimeder(t,'centered');
    
    % Compare
    testCase.verifyEqual(d1dt(2:end-1,:)*Y,dYdt(2:end-1)  ,'AbsTol',sqrt(eps),'First  order time derivatives values from fbttimeder do not match');
    testCase.verifyEqual(d2dt(2:end-1,:)*Y,d2Ydt2(2:end-1),'AbsTol',sqrt(eps),'Second order time derivatives values from fbttimeder do not match');
    
  end
end

end
