classdef doublet_fbt_tests < meq_test
  % Tests involving doublet cases of FBT
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  properties
    doprint = false;
    verbosity = 1;
    shot = 55331;
    t = 0.2;
  end
  
  methods(Test,TestTags={'fbt'})
    
    function fbt_doublet_meqint_test(testCase)
      % Verify domain integrals
      [ok,msg] = meq_test.check_tok('tcv');
      testCase.assumeTrue(ok,msg);
      [L,LX,LY] = fbt('tcv',testCase.shot);
      
      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),'FBT did not converge');
      tol = 1e-4;
      diag = 'Sum of per-domain values do not match total value for ''%s''';
      testCase.verifyEqual(sum(LY.FtD) ,LY.Ft ,'RelTol',tol,sprintf(diag,'Ft'));
      testCase.verifyEqual(sum(LY.Ft0D),LY.Ft0,'RelTol',tol,sprintf(diag,'Ft0'));
      testCase.verifyEqual(sum(LY.IpD) ,LY.Ip ,'RelTol',tol,sprintf(diag,'Ip'));
      testCase.verifyEqual(sum(LY.VpD) ,LY.Vp ,'RelTol',tol,sprintf(diag,'Vp'));
      testCase.verifyEqual(sum(LY.WkD) ,LY.Wk ,'RelTol',tol,sprintf(diag,'Wk'));
      testCase.verifyEqual(sum(LY.WpD) ,LY.Wp ,'RelTol',tol,sprintf(diag,'Wp'));
      testCase.verifyEqual(sum(LY.WtD) ,LY.Wt ,'RelTol',tol,sprintf(diag,'Wt'));
      testCase.verifyEqual(sum(LY.Wt0D),LY.Wt0,'RelTol',tol,sprintf(diag,'Wt0'));
    end
    
    function fbt_doublet_scan(testCase)
      % scan some FBT equilibria transitioning between doublet and droplet
      [ok,msg] = meq_test.check_tok('tcv');
      testCase.assumeTrue(ok,msg);
      [L,LX] = fbt('tcv',testCase.shot,0.02,'izgrid',true,...
        'fbtagcon',{'Ip','Wk','qA'},...
        'bfct',@bf3pmex,'bfp',false,...
        'iterfrz',30,...
        'iterq',10,...
        'ifield',true,...
        'ivacuum',true,...
        'selx','X');
      LX.bpD = 0.25*[1;1;0];
      
      %% Three FBT Droplet/Doublet solutions
      fac = [0,0.12,0.03];
      titstr={'Droplet','Doublet-Droplet','Doublet'};
      for ii=1:3
        k = ~isnan(LX.gpz); % valid points
        gpz = LX.gpz(k);
        gpz = gpz-sign(gpz).*fac(ii);
        LX.gpz(k) = gpz;
        LX = fbtx(L,LX); % Necessary after manual LX update
        LY = fbtt(L,LX);
        
        % Verify FBT converged
        conv = meq_test.check_convergence(L,LX.t,LY);
        testCase.verifyTrue(conv,sprintf('FBT did not converge for %s case',titstr{ii}));
        if ~conv, continue; end
        % check constraints are satisfied
        testCase.verifyEqual(LX.IpD,LY.IpD,'AbsTol',100*LX.tol*L.Ip0)
        testCase.verifyEqual(LX.WkD,LY.WkD,'AbsTol',100*LX.tol*LY.WN)
        testCase.verifyEqual(LX.qA ,LY.qA ,'AbsTol',100*LX.tol)
       
        %% plot
        if testCase.verbosity
          if ii==1, clf; end
          subplot(1,3,ii)
          meqplotfancy(L,LY,'vacuum',true,'openFieldlineStyle','--','openFieldlineColor','k');
          title(titstr{ii})
        end
      end
      
      if testCase.verbosity && testCase.doprint
        set(gcf,'position',[50 50 800 400]); shg;
        print('-depsc','doublet_droplet_fbt')
      end
      
    end
    
    function fbt_asymmetric_scan(testCase)
      % scan some FBT equilibria transitioning between doublet and droplet
      [ok,msg] = meq_test.check_tok('tcv');
      testCase.assumeTrue(ok,msg);
      [L,LX] = fbt('tcv',testCase.shot,0.02,...
        'izgrid',true,'ivacuum',true,...
        'fbtagcon',{'Ip','Wk','qA'},...
        'bfct',@bf3pmex,'bfp',false,...
        'iterq',10,'iterfrz',30);
      
      LX.bpD = [0.25;0.25;0];
      %% Three droplets at different Ip and zA
      scalgrid = [-0.3 0 0.4]; rscalgrid = [-0.1,0,0.1];
      Ip0 = LX.IpD(1); gpr0 = LX.gpr;
      LX0 = LX;
      for jj = 1:2
        for ii=1:numel(scalgrid)
          LX = LX0;
          switch jj
            case 1
              LX.IpD(1:2) = Ip0*(1+[1;-1]*scalgrid(ii));
            case 2
              LX.gpr(LX.gpz>0) = gpr0(LX.gpz>0)*(1-rscalgrid(ii));
          end
          LX = fbtx(L,LX); % Necessary after manual LX update
          LY = fbtt(L,LX);
          
          % check constraints are satisfied
          conv = meq_test.check_convergence(L,LX.t,LY);
          testCase.verifyTrue(conv,'FBT did not converge');
          if ~conv, continue; end
          testCase.verifyEqual(LX.IpD,LY.IpD,'AbsTol',100*LX.tol*L.Ip0)
          testCase.verifyEqual(LX.WkD,LY.WkD,'RelTol',100*LX.tol*LY.WN)
          testCase.verifyEqual(LX.qA ,LY.qA ,'RelTol',100*LX.tol      )
          
          if testCase.verbosity
            if ii==1 && jj==1, clf; end
            subplot(2,3,ii+(jj-1)*3)
            meqplotfancy(L,LY); hold on;
            title(sprintf('IpD=[%2.0f,%2.0f]kA\nrA=[%2.0f,%2.0f]cm',LY.IpD(1:2)/1e3,LY.rA(1:2)*100),...
              'FontSize',10)
          end
        end
      end
      
      if testCase.verbosity && testCase.doprint
        set(gcf,'position',[50 50 800 400]); shg;
        print('-depsc','asymmetric_doublet')
      end
    end
    
    function fbt_stable_doublet(testCase)
      % construct stable droplet equilibria imposing vacuum dBr/dz values at axis
      [ok,msg] = meq_test.check_tok('tcv');
      testCase.assumeTrue(ok,msg);
      
      % Shot in ffelici PCS space
      shot = 903888; %#ok<*PROP>
      static = 3; % version of the static tree without baffles or port-protection tiles
      P = mgp(shot);
      
      %% Initial FBT problem
      [L,LX] = fbt('tcv',shot,[],...
        'static',3,... % No baffles
        'bfct',@bf3pmex,'bfp',false,...
        'fbtagcon',{'Ip','Wk','qA'},...
        'izgrid',true,'ivacuum',true,...
        'useSQP',true,'iterfrz',30);

      %% Add vacuum dBr/dz constraint
      vrz = L.P.iohfb*0.03; % dBrdz desired

      nt = numel(LX.t);
      rr = [1; 1]*P.rmajo1;
      zz = [1;-1]*P.zmajo1;
      vrr = NaN(2,nt);
      vrz = repmat(vrz,2,nt);
      vzz = NaN(2,nt);
      ve = zeros(2,nt); % Inf to disable, 0 for exact constraint, 0.001 is ok, 0.1 too high to get desired value
      %    fbtgp(LX,r ,z ,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
      LX = fbtgp(LX,rr,zz,0,[],[],[],[],[],[],[],[],[],[],[],vrr,vrz,vzz,ve);
      LX.gpvd(:) = 1;

      LX.bpD = repmat([0;0;0],1,numel(LX.t));
      LX = fbtx(L,LX); % Necessary after manual LX update

      %% Solve FBT
      LY = fbtt(L,LX);
      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),'FBT did not converge with vrz constraint')
      kt = 1;
      LYt = meqxk(LY,kt);
      %% Compute vacuum field derivatives using interpolation
      Fxa = reshape(L.G.Mxa*LYt.Ia,L.nzx,L.nrx);
      inM = qintc([],L.drx,L.dzx);
      [~,~,~,~,Brzi,~,~] = qintmex(L.G.rx,L.G.zx,Fxa,rr(:,kt),zz(:,kt),inM);
      target = vrz(:,kt);
      testCase.verifyEqual(Brzi,target,'RelTol',0.15); % can not expect more for qint
      %%
      if testCase.verbosity
        clf;
        meqplotfancy(L,LYt); hold on;
        contour(L.rx,L.zx,reshape(L.G.Mxa*LYt.Ia,L.nzx,L.nrx),'k--');
        LXt = meqxk(LX,kt); plot(LXt.gpr,LXt.gpz,'ok')
      end
    end
  end
end
