classdef bf_multiD_tests < meq_test
  % Tests for multidimensional basis functions
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  properties
    L,LY;
    bfp,bfct;
    ng
  end
  
  properties (ClassSetupParameter)
    bfcts = {@bfgenD, @bfdoublet, @bfgenD};
    bfps = {
     bfpgenD(   {@bf3pmex,false,[1;0;0];@bf3pmex,false,[0;1;0];@bf3pmex,false,[1;1;1]},3), ...
     bfpdoublet({@bf3pmex,false        ;@bf3pmex,false        ;@bf3pmex,false        }  ),...
     bfpgenD(   {@bf3pmex,false,[1;0;0];@bfabmex,[0 0],[0;1;0];@bfabmex,[1 2],[1;1;1]},3); % one empty bf set
     };
  end
  
  methods (TestClassSetup,ParameterCombination='sequential')
    function setupLLY(testCase)
      [L,~,LY] = fbt('ana',82,[]); %#ok<*PROP>
      testCase.L  = L;
      testCase.LY = LY;
    end
    
    function setupbf(testCase,bfcts,bfps)
      testCase.bfp = bfps;
      testCase.bfct = bfcts;
      
      % get and store number of coefficients
      [fPg,~,~] = testCase.bfct(0,testCase.bfp);
      testCase.ng = numel(fPg);
    end
  end
      
  
  methods (Test, TestTags = {'Unit','bf'})
    %% Verify that each mode runs and yields expected size of outputs
    function check_mode_0(testCase)
      bfct = testCase.bfct; bfp = testCase.bfp;
      [fPg,fTg,TDg] = bfct(0,bfp);
      
      testCase.verifyEqual(size(fPg),[testCase.ng,1]);
      testCase.verifyEqual(size(fTg),[testCase.ng,1]);
      testCase.verifyEqual(size(TDg),[testCase.L.nD,testCase.ng]);
    end
    
    function check_mode_1(testCase)
      [L,LY] = deal(testCase.L,testCase.LY);
      bfct = testCase.bfct; bfp = testCase.bfp;
      
      [Tyg,TpDg,ITpDg] = bfct(1,bfp,LY.Fx,LY.F0,LY.F1,LY.Opy,L.ry,L.iry);
      
      testCase.verifyEqual(size(  Tyg),[L.ny,testCase.ng]);
      testCase.verifyEqual(size( TpDg),[L.nD,testCase.ng]);
      testCase.verifyEqual(size(ITpDg),[L.nD,testCase.ng]);
    end
    
    function check_mode_2(testCase)
      [L,LY] = deal(testCase.L,testCase.LY);
      bfct = testCase.bfct; bfp = testCase.bfp;
      
      [gQDg,IgQDg] = bfct(2,bfp,L.pQ.^2,LY.F0,LY.F1);
      
      if L.nD > 1, size_ = [L.nQ,L.nD,testCase.ng];
      else,        size_ = [L.nQ,testCase.ng];
      end
      testCase.verifyEqual(size( gQDg),size_);
      testCase.verifyEqual(size(IgQDg),size_);
    end
    
    function check_mode_3(testCase)
      [L,LY] = deal(testCase.L,testCase.LY);
      bfct = testCase.bfct; bfp = testCase.bfp;
      
      [fPg,fTg,~] = bfct(0,bfp);
      
      [aPpg,aTTpg,aPg,ahqTg] = bfct(3,bfp,ones(testCase.ng,1),LY.F0,LY.F1,fPg,fTg,L.idsx);
      
      testCase.verifyEqual(size( aPpg),[testCase.ng,1]);
      testCase.verifyEqual(size(aTTpg),[testCase.ng,1]);
      testCase.verifyEqual(size(  aPg),[testCase.ng,1]);
      testCase.verifyEqual(size(ahqTg),[testCase.ng,1]);
    end
    
    function check_mode_5(testCase)
      [L,LY] = deal(testCase.L,testCase.LY);
      bfct = testCase.bfct; bfp = testCase.bfp;
      
      [g0g,Ig0g] = bfct(5,bfp,[],LY.F0,LY.F1);
      
      testCase.verifyEqual(size( g0g),[L.nD,testCase.ng]);
      testCase.verifyEqual(size(Ig0g),[L.nD,testCase.ng]);
    end
    
    function check_mode_6(testCase)
      [L,LY] = deal(testCase.L,testCase.LY);
      bfct = testCase.bfct; bfp = testCase.bfp;
      
      r0  = [   LY.rA;zeros(L.nD-LY.nA,1)];
      ir0 = [1./LY.rA;zeros(L.nD-LY.nA,1)];
      
      [Qqg,Xq] = bfct(6,bfp,[],LY.F0,LY.F1,r0,ir0,L.idsx);
      
      testCase.verifyEqual(size(Qqg,2),testCase.ng);
      testCase.verifyEqual(size(Xq,1),size(Qqg,1));
    end
    
    function check_mode_7(testCase)
      [~,LY] = deal(testCase.L,testCase.LY);
      bfct = testCase.bfct; bfp = testCase.bfp;
            
      [Qcg,Xc] = bfct(7,bfp,[],LY.F0,LY.F1);
      
      testCase.verifyEqual(size(Qcg,2),testCase.ng);
      testCase.verifyEqual(size(Xc,1),size(Qcg,1));
    end
    
    %% check continuity across separatrix of basis functions
    function bf_continuity_test(testCase)
      [L,LY] = deal(testCase.L,testCase.LY);
      bfct = testCase.bfct; bfp = testCase.bfp;
            
      [gQDg,IgQDg] = bfct(2,bfp,L.pQ.^2,LY.F0,LY.F1);
      
      testCase.verifyEqual( gQDg(end,1,:), gQDg(1,3,:),'bf functions are discontinuous across separatrix 1->3');
      testCase.verifyEqual(IgQDg(end,1,:),IgQDg(1,3,:),'bf integrals are discontinuous across separatrix 1->3');
      testCase.verifyEqual( gQDg(end,2,:), gQDg(1,3,:),'bf functions are discontinuous across separatrix 2->3');
      testCase.verifyEqual(IgQDg(end,2,:),IgQDg(1,3,:),'bf integrals are discontinuous across separatrix 2->3');
      
    end
  end
  
end
