classdef bf_comp_tests < meq_test
  % Compare bf sets implementations
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  properties
    reltol = sqrt(eps);
    abstol = 10*eps;
    verbosity = 0;
    n = 41; % number of FN points

    % parameters for dummy test equilibrium
    r0 = 1;
    rBt = 1.4;
    
    FN; L; LY;
  end

  properties (Abstract)
    bf1
    bf2
  end
  
  properties (ClassSetupParameter)
    % Various combinations of boundary and axis flux
    FAs    = {-1   0   2  0.5 };
    FBs    = {0    1   0  0   };
  end
  
  properties (TestParameter,Abstract)
    bfp1
    bfp2
  end
  
  methods(TestClassSetup,ParameterCombination='sequential')
    
    function gen_Fx(testCase,FAs,FBs)
      r0 = testCase.r0; z0=0; rBt = testCase.rBt; %#ok<*PROPLC>
      L = liu('ana',1);
      
      [testCase.L,testCase.LY] = testCase.getCircularEquilibrium(L,r0,z0,FAs,FBs,rBt);
      testCase.FN = linspace(0,1,testCase.n)';
     end
    
  end

  methods(Test, ParameterCombination='sequential', TestTags = {'Unit','bf'})
   
    function test_mode0(testCase,bfp1,bfp2)
      % Test 
      [fPg ,fTg ] = feval(testCase.bf1,0,bfp1);
      [fPg_,fTg_] = feval(testCase.bf2,0,bfp2);
      testCase.check_tol(fPg,fPg_,'mismatch for fPg');
      testCase.check_tol(fTg,fTg_,'mismatch for fTg');
    end
    
    function test_mode1(testCase,bfp1,bfp2)
      % Test evaluating basis function and its integrals on rz grid
      L  = testCase.L ;
      LY = testCase.LY;
      [Tyg ,TpDg ,ITpDg ] = feval(testCase.bf1,1,bfp1, ...
        LY.Fx,LY.FA,LY.FB,LY.Opy,L.ry,L.iry);
      [Tyg_,TpDg_,ITpDg_] = feval(testCase.bf2,1,bfp2, ...
        LY.Fx,LY.FA,LY.FB,LY.Opy,L.ry,L.iry);
      
      % Verify match
      testCase.check_tol(Tyg  , Tyg_  , 'Tyg'  )
      testCase.check_tol(TpDg , TpDg_ , 'TpDg' )
      testCase.check_tol(ITpDg, ITpDg_, 'ITpDg')
      
    end
    
    function test_mode2(testCase,bfp1,bfp2)
      % Test evaluating basis function and its integrals on FN grid
      LY = testCase.LY;
      args = {testCase.FN,LY.FA,LY.FB}; % arguments common to all bfs
      [gN ,IgN ] = feval(testCase.bf1,2,bfp1,args{:});
      [gN_,IgN_] = feval(testCase.bf2,2,bfp2,args{:});
      
      % Verify match
      testCase.check_tol( gN,  gN_, 'gN' )
      testCase.check_tol(IgN, IgN_, 'IgN')
    end
    
    function test_mode3(testCase,bfp1,bfp2)
      % Test transformation of coefficients for profile evaluation
      L  = testCase.L ;
      LY = testCase.LY;
      [fPg,fTg] = feval(testCase.bf1,0,bfp1);
      ag0 = ones(size(fPg)); % unit coefficients just to get scaling
      args = {ag0,LY.FA,LY.FB,fPg,fTg,L.idsx}; % arguments common to all bfs
      
      [aPpg ,aTTpg ,aPg ,ahqTg ] = feval(testCase.bf1,3,bfp1,args{:});
      [aPpg_,aTTpg_,aPg_,ahqTg_] = feval(testCase.bf2,3,bfp2,args{:});
      
      % Verify match
      testCase.check_tol(aPpg , aPpg_ , 'aPpg' )
      testCase.check_tol(aTTpg, aTTpg_, 'aTTpg')
      testCase.check_tol(aPg  , aPg_  , 'aPg'  )
      testCase.check_tol(ahqTg, ahqTg_, 'ahqTg')
    end
   
    function test_mode4(testCase,bfp1,bfp2)
      % Test computation of vertical integrals
      LY = testCase.LY;
      kd = uint32(0);
      fd = [1;0];
      args = {LY.Fx,LY.FA,LY.FB,LY.Opy,kd,fd}; % arguments common to all bfs
      [Tdg ,Tgy ] = feval(testCase.bf1,4,bfp1,args{:});
      [Tdg_,Tgy_] = feval(testCase.bf2,4,bfp2,args{:});
      
      % Verify match
      testCase.check_tol(Tdg, Tdg_, 'Tdg')
      testCase.check_tol(Tgy, Tgy_, 'Tgy')
    end
    
    function test_mode5(testCase,bfp1,bfp2)
      % Test axis values
      LY = testCase.LY;
      args = {[],LY.FA,LY.FB};
      [GA ,IGA ] = feval(testCase.bf1,5,bfp1,args{:});
      [GA_,IGA_] = feval(testCase.bf2,5,bfp2,args{:});
      
      % Verify match
      testCase.check_tol(GA , GA_ , 'GA' )
      testCase.check_tol(IGA, IGA_, 'IGA')
    end
   
    function test_mode6(testCase,bfp1,bfp2)
      % Test computation of elements for the regularisation equations
      L  = testCase.L ;
      LY = testCase.LY;
      rA = testCase.r0;
      args = {[],LY.FA,LY.FB,rA,1./rA,L.dsx}; % arguments common to all bfs
      [Qqg ,Xq ] = feval(testCase.bf1,6,bfp1,args{:});
      [Qqg_,Xq_] = feval(testCase.bf2,6,bfp2,args{:});
      
      % Verify match
      testCase.check_tol(Qqg, Qqg_, 'Qqg')
      testCase.check_tol(Xq , Xq_ , 'Xq' )
    end
   
    function test_mode7(testCase,bfp1,bfp2)
      % Test computation of elements for the inequality constraints
      LY = testCase.LY;
      args = {[],LY.FA,LY.FB}; % arguments common to all bfs
      [Qcg ,Xc ] = feval(testCase.bf1,7,bfp1,args{:});
      [Qcg_,Xc_] = feval(testCase.bf2,7,bfp2,args{:});
      
      % Verify match
      testCase.check_tol(Qcg, Qcg_, 'Qcg')
      testCase.check_tol(Xc , Xc_ , 'Xc' )
    end
   
    function test_mode8(testCase,bfp1,bfp2)
      % Test computation of toroidal magnetic field
      L  = testCase.L ;
      LY = testCase.LY;
      [fPg,~] = feval(testCase.bf1,0,bfp1);
      ag = 1e4*rand(size(fPg)); % random coefficients for tests
      args = {LY.Fx,LY.FA,LY.FB,LY.Opy,ag,LY.rBt,L.idsx,L.iry}; % arguments common to all bfs
      [Bty ] = feval(testCase.bf1,8,bfp1,args{:});
      [Bty_] = feval(testCase.bf2,8,bfp2,args{:});
      
      % Verify match
      testCase.check_tol(Bty, Bty_, 'Bty')
    end
    
    function test_mode11(testCase,bfp1,bfp2)
      % Test evaluating basis function and its integrals on rz grid
      L  = testCase.L ;
      LY = testCase.LY;
      [dTygdFy ,dTygdFA ,dTygdFB, dITygdFA ,dITygdFB ] = feval(testCase.bf1,11, ...
        bfp1,LY.Fx,LY.FA,LY.FB,LY.Opy,L.ry,L.iry);
      [dTygdFy_,dTygdFA_,dTygdFB_,dITygdFA_,dITygdFB_] = feval(testCase.bf2,11, ...
        bfp2,LY.Fx,LY.FA,LY.FB,LY.Opy,L.ry,L.iry);
      
      % Verify match
      testCase.check_tol( dTygdFy,  dTygdFy_,  'dTygdFy')
      testCase.check_tol( dTygdFA,  dTygdFA_,  'dTygdFA')
      testCase.check_tol( dTygdFB,  dTygdFB_,  'dTygdFB')
      testCase.check_tol(dITygdFA, dITygdFA_, 'dITygdFA')
      testCase.check_tol(dITygdFB, dITygdFB_, 'dITygdFB')
    end
    
    function test_mode12(testCase,bfp1,bfp2)
      % Test evaluating basis function and its integrals on FN grid
      LY = testCase.LY;
      args = {testCase.FN,LY.FA,LY.FB}; % arguments common to all bfs
      [dgNdFN ,dgNdF0 ,dgNdF1 ,dIgNdFN ,dIgNdF0 ,dIgNdF1 ] = feval(testCase.bf1,12,bfp1,args{:});
      [dgNdFN_,dgNdF0_,dgNdF1_,dIgNdFN_,dIgNdF0_,dIgNdF1_] = feval(testCase.bf2,12,bfp2,args{:});
      
      % Verify match
      testCase.check_tol( dgNdFN,  dgNdFN_,  'dgNdFN')
      testCase.check_tol( dgNdF0,  dgNdF0_,  'dgNdF0')
      testCase.check_tol( dgNdF1,  dgNdF1_,  'dgNdF1')
      testCase.check_tol(dIgNdFN, dIgNdFN_, 'dIgNdFN')
      testCase.check_tol(dIgNdF0, dIgNdF0_, 'dIgNdF0')
      testCase.check_tol(dIgNdF1, dIgNdF1_, 'dIgNdF1')
      
    end
    
    function test_mode13(testCase,bfp1,bfp2)
      % Test transformation of coefficients for profile evaluation
      L  = testCase.L ;
      LY = testCase.LY;
      [fPg,fTg] = feval(testCase.bf1,0,bfp1);
      ag0 = ones(size(fPg)); % unit coefficients just to get scaling
      args = {ag0,LY.FA,LY.FB,fPg,fTg,L.idsx}; % arguments common to all bfs
      
      [daPpgdF0 ,daTTpgdF0 ,daPgdF0 ,dahqTgdF0 ,daPpgdF1 ,daTTpgdF1 ,daPgdF1 ,dahqTgdF1 ] = ...
        feval(testCase.bf1,13,bfp1,args{:});
      [daPpgdF0_,daTTpgdF0_,daPgdF0_,dahqTgdF0_,daPpgdF1_,daTTpgdF1_,daPgdF1_,dahqTgdF1_] = ...
        feval(testCase.bf2,13,bfp2,args{:});
      
      % Verify match
      testCase.check_tol(daPpgdF0 , daPpgdF0_ , 'daPpgdF0' )
      testCase.check_tol(daTTpgdF0, daTTpgdF0_, 'daTTpgdF0')
      testCase.check_tol(daPgdF0  , daPgdF0_  , 'daPgdF0'  )
      testCase.check_tol(dahqTgdF0, dahqTgdF0_, 'dahqTgdF0')
      testCase.check_tol(daPpgdF1 , daPpgdF1_ , 'daPpgdF1' )
      testCase.check_tol(daTTpgdF1, daTTpgdF1_, 'daTTpgdF1')
      testCase.check_tol(daPgdF1  , daPgdF1_  , 'daPgdF1'  )
      testCase.check_tol(dahqTgdF1, dahqTgdF1_, 'dahqTgdF1')
    end
    
    function test_mode15(testCase,bfp1,bfp2)
      % Test axis values
      LY = testCase.LY;
      args = {[],LY.FA,LY.FB};
      [dgAdFBA ,dIgAdFBA ] = feval(testCase.bf1,15,bfp1,args{:});
      [dgAdFBA_,dIgAdFBA_] = feval(testCase.bf2,15,bfp2,args{:});
      
      % Verify match
      testCase.check_tol( dgAdFBA, dgAdFBA_,  'dgAdFBA');
      testCase.check_tol(dIgAdFBA,dIgAdFBA_, 'dIgAdFBA');
    end
   
    function test_mode16(testCase,bfp1,bfp2)
      % Test computation of derivatives of elements for the regularisation equations
      L  = testCase.L ;
      LY = testCase.LY;
      rA = testCase.r0;
      args = {[],LY.FA,LY.FB,rA,1./rA,L.dsx}; % arguments common to all bfs
      [dQqgdF0 ,dQqgdF1 ,dQqgdr0 ,dQqgdir0 ] = feval(testCase.bf1,16,bfp1,args{:});
      [dQqgdF0_,dQqgdF1_,dQqgdr0_,dQqgdir0_] = feval(testCase.bf2,16,bfp2,args{:});
      
      % Verify match
      testCase.check_tol(dQqgdF0 , dQqgdF0_ , 'dQqgdF0' )
      testCase.check_tol(dQqgdF1 , dQqgdF1_ , 'dQqgdF1' )
      testCase.check_tol(dQqgdr0 , dQqgdr0_ , 'dQqgdr0' )
      testCase.check_tol(dQqgdir0, dQqgdir0_, 'dQqgdir0')
    end
   
    function test_mode17(testCase,bfp1,bfp2)
      % Test computation of derivatives of elements for the inequality constraints
      L  = testCase.L ;
      LY = testCase.LY;
      args = {[],LY.FA,LY.FB}; % arguments common to all bfs
      [dQcgdF0 ,dQcgdF1 ] = feval(testCase.bf1,17,bfp1,args{:});
      [dQcgdF0_,dQcgdF1_] = feval(testCase.bf2,17,bfp2,args{:});
      
      % Verify match
      testCase.check_tol(dQcgdF0, dQcgdF0_, 'dQcgdF0')
      testCase.check_tol(dQcgdF1, dQcgdF1_, 'dQcgdF1')
    end
  end

  % Helper methods
  methods
    function check_tol(testCase,value,expected,name)
      % Verify relative or absolute tolerance check is verified
      testCase.verifyEqual( value, expected,'AbsTol',testCase.abstol,'RelTol',testCase.reltol,sprintf('mismatch for %s',name));
    end
  end
end
