classdef bdet_test < meq_test
  % Tests for bdet circuit simulator
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

  properties
    verbosity = 0;
  end

  properties(TestParameter)
    tgrid_regular = {true,false}
    vacuum_only = {true,false}
  end

  methods(Test, TestTags={'fge'} )
    function test_vs_forward_circuit_equation(testCase,tgrid_regular,vacuum_only)
      % test that (Ia,Iy)->bdet->(Iu,Va)
      % gives same result as (Va,Iy)->circuit equation->(Ia,Iu)
      
      % Setup
      [L,LX1] = fge('ana',91,0,'selu','e','nu',30);

      %%
      nt = 11; dt = 1e-3;
      if tgrid_regular
        t = dt*(0:(nt-1));
      else
        t = cumtrapz(dt*(0.5+rand(1,nt)));
      end
      LX = fgex('ana',t,L,LX1);

      % override Ia
      LX.Ia = 100*rand(L.G.na,nt);

      if vacuum_only
        LX.Iy = zeros(L.nzy,L.nry,nt);
      else
        LX.Iy = LX1.Iy.*(1+0.05*rand(1,1,nt));
      end
      LX.Ip = squeeze(sum(LX.Iy,[1,2]))';

      % solve bdet
      LY = bdet(L,LX,'debug',1,'debugplot',2); % always test debug text and plot as well

      if vacuum_only
        testCase.assumeTrue(tgrid_regular,'skipping fget case for irregular time grid')
        %% Forward solve FGE in vacuum case
        LX.Va = LY.Va; % assign input Va from bdet output;
        LX.Ia(:,1) = LY.Ia(:,1); % initial Ia;

        LYf = fget(L,LX);
        Ia = LYf.Ia; Iu = LYf.Iu; % for comparison
      else
        %% Forward solve circuit equation for Ia0,Iu0 with this Va, Iy
        dIy = diff(LY.Iy,[],3);
        Ia0 = LY.Ia(:,1);
        Iu0 = LY.Iu(:,1);

        Mee = [L.G.Maa,L.G.Mau;L.G.Mau',L.G.Muu];
        Ree = diag([L.G.Ra;L.G.Ru]);

        % Mee*Iedot + Re*Ie + Mey*Iydot = Va
        Ie = zeros(L.ne,numel(LX.t));
        Ie(:,1) = [Ia0;Iu0];

        for it=2:nt
          dt = LX.t(it)-LX.t(it-1);
          % Implicit Euler method discretization
          % Mee*(Iek-Iep) + dt*Re*Iek + dt*Mey*Iydot = dt*Va
          % (Mee+dt*Re)*Iek = Mee*Iep -    Mey*dIy + dt*Va
          % Iek =  (Mee+dt*Re) \ (Mee*Iep - Mey*dIy+ dt*Va)
          MM = (Mee + dt*Ree); Ba = eye(L.ne,L.G.na);
          Ie(:,it) = MM\(Mee*Ie(:,it-1) + dt*Ba*LY.Va(:,it) - L.Mey*reshape(dIy(:,:,it-1),L.ny,1));
        end
        Ia = Ie(1:L.G.na,:); Iu = Ie(L.G.na + (1:L.G.nu),:);

      end

      tol = 10*eps;
      testCase.verifyEqual(LY.Ia,Ia,'AbsTol',tol*L.Ia0)
      testCase.verifyEqual(LY.Iu,Iu,'AbsTol',tol*L.Iu0)

    end
  end
end