classdef asxycs_jacobian_test < meq_jacobian_test
  % Test the jacobian of asxy with cubic spline interpolation
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
    
  properties
    verbosity = 0;
    tok = 'ana';
  end
    
  properties(ClassSetupParameter)
    shot = struct('circular',1,'diverted',2,'diverted2',3,...
      'double_null',4,'squashed',5,...
      meq_jacobian_test.doublets_under_test{:});
  end

  properties (TestParameter)
    convergence_test = struct('false',false,'true',true);
  end
  
  methods(TestClassSetup, ParameterCombination='sequential')
    function setup(testCase,shot)
      % Compute FGS initial guess (FBT solution)
      t = 0;
      testCase.get_fgs_LLX(shot,t);
      testCase.L.P.icsint = true;
      testCase.L.P.ilim   = 3;
      testCase.L = fgsc(testCase.L.P,testCase.L.G);
      if (shot == 2)
        testCase.meshDeltaFx = logspace(-5,-4,3);
      end
    end
  end
      
  methods(Test,TestTags={'Jacobian'})
    function test_extremum_jacobian(testCase,convergence_test)

      L = testCase.L;
      Fx = testCase.LX.Fx;

      % Prepare arguments
      fun = @asxymex_plus_asxycs;
      x0 = {Fx,L};

      % Get locations
      [zA,rA,~,dz2FA,dr2FA,drzFA,zX,rX,~,dz2FX,dr2FX,drzFX] = ...
        asxymex_plus_asxycs(Fx,L);

      % Check we aren't too close to cell boundaries
      mindz = min(abs([zA;zX] - L.zx.'),[],[1,2]);
      mindr = min(abs([rA;rX] - L.rx.'),[],[1,2]);
      testCase.assumeTrue(min(mindr,mindz)>1e-4,'Skipping test as singular point is too close to grid cell boundary');

      % Check position, flux and hessian
      names_out = {'zA','rA','FA','dz2FA','dr2FA','drzFA',...
                   'zX','rX','FX','dz2FX','dr2FX','drzFX'};
      iargout = 1:12;
      names_in = {'Fx'};
      iargin = 1;
      
      % Get jacobians
      J0 = cell(12,1);
      [J0{[3,1,2,4,5,6]  }] = asxycsJac(Fx,zA,rA,dz2FA,dr2FA,drzFA,L);
      [J0{[3,1,2,4,5,6]+6}] = asxycsJac(Fx,zX,rX,dz2FX,dr2FX,drzFX,L);
      
      % F0 sizes
      nA = numel(zA);
      nX = numel(zX);
      szF0 = [repmat({nA},1,6),repmat({nX},1,6)];
      jacfd_args = {'szF0',szF0};
      
      % Skip tests for X-point hessian (convergence range for
      % HA and HX seem to not be connected)
      mask = 1:9;
      names_out = names_out(mask);
      iargout = iargout(mask);
      J0 = J0(mask,:);

      if convergence_test
        % Test convergence
        epsval = {testCase.meshDeltaFx};
      else
        % Test baseline error
        epsval = {testCase.deltaFx};
      end
      % Call generic function
      testCase.test_analytical_jacobian(...
        fun,x0,J0,jacfd_args,iargout,names_out,iargin,names_in,epsval);
    end
  end
end

function [zA,rA,FA,dz2FA,dr2FA,drzFA,zX,rX,FX,dz2FX,dr2FX,drzFX] = ...
        asxymex_plus_asxycs(Fx,L)

% Call asxy
[zA,rA,~,~,~,~,~,zX,rX,~,~,~,~,~] = ...
  asxymex(Fx,L.G.zx,L.G.rx,L.P.dasm,L.dzx,L.drx,L.idzx,L.idrx,L.Oasx,L.dimw);

% Call asxycs
[zA,rA,FA,dz2FA,dr2FA,drzFA,zX,rX,FX,dz2FX,dr2FX,drzFX] = ...
  asxycs(Fx,zA,rA,zX,rX,L);

end
