classdef LY2SC_test < meq_test
  % LY to shape control points test.
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  %%
  properties
    verbosity = true; % Verbosity level
    tol = 1e-3; % Tolerance on control points positioning [m] and on interpolated flux [Wb] and field [T]
  end
  properties(TestParameter)               
    r0  = {0.85};  % Centroid coordinates for circular equilibrium generation
    z0  = {0};
    nc  = {10};
    ncC = {7};
  end
  
  %%
  methods(Test, TestTags = {'Unit'})
    function test_circular_map(testCase,r0,z0,nc,ncC)
      
      % generate circular flux map for anamak
      L = fbt('ana',1,0);
      [L,LY] = meq_test.getCircularEquilibrium(L,r0,z0,1,0,1);
      LY.t = 0; LY.rX=[]; LY.zX=[]; LY.FX=[];            
      
      % get contour points
      SC = LY2SC(L,LY,nc);
      
      % compute radius for each control point
      radius = abs(r0-LY.rB);
      SCradii = sqrt((SC.rc-r0).^2 + (SC.zc-z0).^2);
      
      % verify the points are on a circle
      testCase.verifyEqual(SCradii,repmat(radius,nc,1),'AbsTol',testCase.tol);
             
      % repeat with less contour points (this validates other if branches)      
      SC2 = LY2SC(L,LY,nc,ncC);
      SC2radii = sqrt((SC2.rc(1:ncC)-r0).^2 + (SC2.zc(1:ncC)-z0).^2);
      testCase.verifyEqual(SC2radii,repmat(radius,ncC,1),'AbsTol',testCase.tol);
      
      % repeat with ncC>nc (ncC gets overwritten in this case)     
      SC3 = LY2SC(L,LY,nc,nc+1);
      SC3radii = sqrt((SC3.rc-r0).^2 + (SC3.zc-z0).^2);
      testCase.verifyEqual(SC3radii,repmat(radius,nc,1),'AbsTol',testCase.tol);
      
      if testCase.verbosity
        figure
        SCplot(SC,L,LY)
        hold on
        SCplot(SC2,L,[],'color','r')
        SCplot(SC3,L,[],'color','b')
      end               
    end    
      
    
    function test_anamak_divertor_map(testCase,nc)
      
      % generate circular flux map for anamak
      [L,~,LY] = fbt('ana',3,0);     
      
      % get contour points
      SC = LY2SC(L,LY,nc);
      
      % interpolate fluxes on resulting points
      inp = struct('n',9);
      qpM = qintc(inp,L.drx,L.dzx);
      [FI,BRI,BZI] = qintmex(L.rx,L.zx,LY.Fx,SC.rc,SC.zc,qpM);

      % Check flux at contour and strike points
      testCase.verifyEqual(FI(SC.lcC|SC.lcL|SC.lcX1|SC.lcS),repmat(LY.FB,sum(SC.lcC|SC.lcL|SC.lcX1|SC.lcS),1),'AbsTol',testCase.tol);
      
      % Check field at X-points
      testCase.verifyEqual(BRI(SC.lcX1|SC.lcX2),zeros(sum(SC.lcX1|SC.lcX2),1),'AbsTol',testCase.tol);
      testCase.verifyEqual(BZI(SC.lcX1|SC.lcX2),zeros(sum(SC.lcX1|SC.lcX2),1),'AbsTol',testCase.tol);

      if testCase.verbosity  
        figure
        SCplot(SC,L,LY)
      end      
    end 
  end    
end
