function [x, solverinfo] = solveF(F,x0,varargin)
% function [x, solverinfo] = solveF(F,x0,varargin)
% General solver to find solution of F(x)=0
% Implements jacobian free and jacobian based Newton Krylov methods
% Main implemented algorithms:
%   - matlab native gmres
%   - direct inversion using J\b
%   - gmres implementations aya, sim, giv (see gmres_jfnk() for more details)
%
% inputs:
%   F:          Function handle, to solve F==0 for. For requirements, see below
%   x0:         Vector, inital solution guess
%   varargin:   struct, solver parameters. For more info, see %% Parse inputs below
% returns:
%   x:          Vector, Solution, if solver converged, otherwise its set to x0
%   solverinfo: Struct, Info struct for solver operation
%
% Requirements for F:
%   + the function signature should be
%     [value,post,jacx,jacu,rowmask] = F(x,opts)
%   + For inputs:
%     - x is a vector containing the current input values
%     - opts is a structure with fields defined in optsF.m
%     'dojacu' with logical values which are requests that the
%     corresponding outputs be computed
%   + For outputs
%     - value is a vector containing the values of the function F at x
%     - post is some post-processing of the current F evaluation, there are
%     no requirements for its class or content
%     - jacx is the jacobian matrix of F with respect to its input x
%     - jacu is the jacobian matrix of F with respect to external
%     parameters u
%     - rowmask is a vector with as many elements as rows in jacx such that
%     if rowmask(i) is non-zero then jacx(i,:) is a row with a single
%     non-zero element at position i whose value is rowmask(i), meaning
%     that jacx(rowmask,rowmask) is a diagonal matrix and
%     jacx(rowmask,~rowmask) is a zero matrix
%
% Debugging levels ('debug' parameter):
%    >0: Display final solver results
%    >3: Display full input parameter structure
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

%% Parse inputs
ip = inputParser; % initialize parser
ip.addParameter('tolF',1e-10); % default value of tolF (minimum tolerance max(abs(F(x))))
ip.addParameter('kmax',50); % default value of kmax (maximum number of Newton steps)
ip.addParameter('algoF','jfnk'); % 'jfnk' 'newton' 'broyden' Specify the algorithm to be used
ip.addParameter('anajac',false); % Use semi-analytical jacobian
ip.addParameter('debug',0);     % Level of solving display verbosity.
ip.addParameter('debugplot',0); % Level of solver plotting verbosity.
ip.addParameter('debugfail',0); % Flag to trigger extra plotting upon failed Newton steps
ip.addParameter('debuglinesrch',0); % Flag to trigger extra plotting of line search space
ip.addParameter('workers',0); % number of workers for parallel computation
ip.addParameter('userowmask',true); % Reduce system if jacobian is known to be block diagonal
ip.addParameter('relax',0); % Apply relaxation on solution
ip.addParameter('jacobian_handle',false); % use function handle form of jacobian
ip.addParameter('nrestarts',0); % number of restarts of newton iteration
ip.addParameter('restart_random',true); % randomized Newton solver restarts
ip.addParameter('random_tries',10); % Number of tries per randomized Newton solver restarts
ip.addParameter('restart_iter_frac',1/5); % fraction of iterations left before adding more upon restart
ip.addParameter('mkryl',200); % default value of m (size of Krylov subspace)
ip.addParameter('epsilon_d',sqrt(eps)); % default values of epsilon_d (variation used to approximate the Jacobian)
ip.addParameter('prectype','user'); % type of preconditioner, see fgsp
ip.addParameter('precupdate','once'); % frequency at which to update preconditioner, see fgsp
ip.addParameter('Prec',[]); % pre-supplied preconditioner
ip.addParameter('ilu_algo','crout'); % ilu algo to use if prectype=='ilu_matlab'. See matlab ilu for more info
ip.addParameter('ilu_droptol',1e-3); % ilu droptol to use if prectype=='ilu_matlab'. See matlab ilu for more info
ip.addParameter('robust_on',1); % flag if finite difference step size should be optimized
ip.addParameter('epsilon_d_tol',10*eps); % min value of epsilon_d when robust decrease on
% 'sim' = solve at each iteration
% 'giv' = use Givens rotations to make H upper triangular
% 'aya' = Ayachour's method to solve H\G iteratively
ip.addParameter('algoGMRES','giv');

% Broyden solver parameters
ip.addParameter('use_inverse',true); % Whether to approximate the Jacobian or the inverse Jacobian in Broyden's method
ip.addParameter('large_system',true);  % Parameter of broyden: do not udpate the inverse matrix at each iteration, but only at the end
ip.addParameter('is_factored',true); % Parameter of broyden: Keep a low rank approximation of the inverse matrix
ip.addParameter('ldim',10); % Rank of the inverse matrix approximation. Only used if is_factored=true
ip.addParameter('iy',[]);   % Idx providing corresponding to the plasma domain. Only used when the inverse jacobian
                            % is approximated by a low rank approx (only if is_factored=true)
ip.addParameter('group_size',4); % Update of rank group_size of the inverse
ip.addParameter('epsilon_res',1e-12); % Residual tolerance for inner residual of MGS methods.
ip.addParameter('prev_H',[]);
ip.addParameter('ratio_gb',0.5);

parse(ip,varargin{:}); % parse the inputs, input xyz is now available in P.xyz
P = ip.Results;

% Debug levels
dispsummary   = (P.debug>1); % display summary of final solver results
dispinputs    = (P.debug>3); % display parameters used as inputs in solveF call

if dispinputs
  fprintf('Call to solveF with these parameters:\n')
  dispparams(P)
end

% supress gmres warning (although it might be good for performance to use
% a higher tolerance for matlab gmres)
w = warning('off', 'MATLAB:gmres:tooSmallTolerance');
tstart=tic;
if isequal(P.algoF,'broyden')
  [x,solverinfo] = broyden_solver(F,x0,P      );
else
  [x,solverinfo] =  newton_solver(F,x0,P,false);
end
% switch warning back
warning(w);

% display what happened
if dispsummary
  if solverinfo.isconverged
    fprintf( ' Solver (%s method) converged in %d iterations with %d F evaluations. ',...
      P.algoF,solverinfo.niter, solverinfo.nfeval);
  elseif ~any(isnan(x))
    fprintf( ' Solver (%s method) did not converge. ', P.algoF );
  else
    warning('No valid solution found using algo %s\n', P.algoF)
  end
  fprintf(' Solver time: %f[s]\n',toc(tstart));
end

end