function Pinv = preconditioner_init(P, J, nx)
% Pinv = preconditioner_init(P, J, nx)
% Returns the preconditioner depending on the preconditioner mode
% and the current system given by J.
%
% A function handle computing v->inv(M) * v is returned. Given P.Prec,
% there are the following options:
%   - P.Prec is a scalar. In this case inv(M) = Id * P.Prec is assumed
%     P.Prec = 1 is used for no preconditioner
%   - P.Prec is a matrix . In this case inv(M) = P.Prec is assumed
%   - P.Prec is a equal to 'ilu_matlab' In this case the resulting function
%     handle computes the squential inversion of the triangular matrices
%     returned by the incomplete LU factorization. In this case, the
%     parameters P.ilu_algo and P.ilu_droptol can be used to tailor the ilu
%     factorization (see matlab ilu for more info)
%
% If the option algoGMRES is set to "direct_inversion", Pinv is left empty as
% no preconditioner is required in this case
%
% inputs:
%   P:        struct, parameter structure for the solver
%   J:        function handle or (Sparse) Matrix, Jacobian. Is only used
%             when its given as a matrix (for ilu preconditioning)
% returns:
%   Pinv:     function handle, function computing v->inv(M) * v for
%             the preconditioning matrix M
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% no preconditioner needed in this case
if strcmp(P.algoGMRES, 'direct_inversion') || ...
    (strcmp(P.prectype,'user') && isempty(P.Prec))
  Pinv = [];
  return;
end

% Check Jacobian size if needed
if ~strcmp(P.prectype,'user') && ~(isnumeric(J) && isequal(size(J),[nx,nx]))
  error('Need a [%dx%d] matrix for J when prectype=%s',nx,nx,P.prectype)
end

switch P.prectype
  case 'user'
    if isa(P.Prec,'function_handle')
      Pinv = P.Prec;
    elseif isnumeric(P.Prec)
      % if P.Prec is empty, use identity
      % if P.Prec is a scalar s, M = Id * s^(-1) <=> inv(M) = Id * s is assumed
      % if P.Prec is a matrix, inv(M) = P.Prec is assumed
      if isempty(P.Prec) || isscalar(P.Prec)
        Pinv = P.Prec;
      else
        if ~isequal(size(P.Prec),[nx,nx])
          error('Preconditioner matrix is of incorrect size');
        end
        Pinv = P.Prec;
      end
    else
      error('Preconditioner is of incorrect type, expect function_handle or numeric');
    end
  case 'inv'
    % Compute jacobian inverse and check for singular matrix
    lastwarn(''); % Reset
    Pinv = inv(J);
    [~,wid] = lastwarn;
    if strcmp(wid,'MATLAB:singularMatrix')
      error('Provided jacobian is singular')
    end
  case 'ilu'
    [J_L, J_U] = ilu(sparse(J), struct('type', P.ilu_algo, 'droptol', P.ilu_droptol));
    % In this case, Pinv is the function handle inverting these sparse
    % triangular matrices in sequence. For a sparse triangular matrix,
    % the \ operator automatically selects the optimal solving strategy
    Pinv = @(v_) J_U \ (J_L \ v_);
  otherwise
    error('Not a valid preconditioning method: "%s"', P.prectype);
end
if ~isempty(Pinv) && isnumeric(Pinv) && strcmpi(P.algoGMRES,'matlab_gmres')
  % MATLAB gmres expects function handle or uses @(x)P\x
  Pinv = @(x) Pinv*x;
end
end
