function [dx, gmres_iter] = invert_jacobian(J, b, Pinv, P)
% INVERT_JACOBIAN Solve linear problem J*dx=b using GMRES or directly
%
%   [dx, gmres_iter] = invert_jacobian(J, b, Pinv, P)
%
% inputs:
%   J:          Numeric matrix (possibly sparse) or function handle
%   b:          Vector or matrix, system rhs (b = -F(x) for the newton step)
%   Pinv:       Preconditioner for use in GMRES methods, either a function handle
%               or numeric type (matrix or scalar). Set Pinv=[] for no preconditioner.
%   P:          struct, parameter structure for the whole Newton solver
% returns:
%   dx:         Vector, Solution of J\b (if solver converged)
%   gmres_iter: iter, Number of gmres iterations done (0 for direct_inversion)
%
% The number of evaluations of J is gmres_iter*size(b,2)
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% Check if J is a function handle or a matrix
isJFunctionHandle = isa(J,'function_handle');

% Check if there are multiple right hand sides
[n,nb] = size(b);
isSingleRHS = nb==1;
% if the system is reduced to a size smaller than mkryl, bound mkryl
mkryl = min(P.mkryl, n);

% Check compatibility of algoGMRES
if isJFunctionHandle && isSingleRHS
  GMRES_opts = {'matlab_gmres','sim','giv','aya'};
  descr = 'J as a function handle and 1 RHS';
elseif isSingleRHS
  GMRES_opts = {'direct_inversion','matlab_gmres','sim','giv','aya'};
  descr = 'J as a matrix and 1 RHS';
elseif isJFunctionHandle
  GMRES_opts = {'sim','giv','qr'};
  descr = 'J as a function handle and multiple RHS';
else 
  GMRES_opts = {'direct_inversion','sim','giv','qr'};
  descr = 'J as a matrix and multiple RHS';
end
if ~any(strcmp(P.algoGMRES,GMRES_opts))
  error('invert_jacobian:algoGMRES','algoGMRES option "%s" does not support %s',P.algoGMRES, descr);
end

% Call linear solver
switch P.algoGMRES
  case 'direct_inversion'
    % simplest algorithm possible. Possibly also the slowest
    if isSingleRHS
      dx = J \ b;
    else
      % Use single LU decomposition to treat all RHS at once
      [L,U,P] = lu(J);
      dx = U\(L\(P*b));
    end
    gmres_iter = 0;
  case 'matlab_gmres'
    % using MATLAB's GMRES with J and Pinv function handle for preconditioning
    if isnumeric(Pinv) && isscalar(Pinv)
      Pinv = @(x) Pinv*x;
    end
    [dx, ~, ~, gmres_iter] = gmres(J, b, [], P.epsilon_res, mkryl, Pinv);
    gmres_iter = gmres_iter(2); % only interested in the number of inner iters
  otherwise
    % use meq implementation of GMRES
    if isSingleRHS
      % Use GMRES with single RHS
      [dx, gmres_iter] =  gmres_jfnk(J, b, P.algoGMRES, P.epsilon_res, mkryl, Pinv);
    else
      % Use BlockGMRES for multple RHSs
      [dx, gmres_iter] = gmres_block(J, b, P.algoGMRES, P.epsilon_res, mkryl, Pinv);
    end
end
end
