function [dx, j] = gmres_jfnk(J, b, type, epsilon_res, mkryl, Pinv)
% GMRES_JFNK Implements the GMRES algorithm to solve J*dx = b
%
%   [dx, j] = gmres_jfnk(J, b, type, epsilon_res, Pinv)
%
% Implements three different methods:
%   - 'sim' = solve H\G at each iteration
%   - 'giv' = use Givens rotations to make H upper triangular
%   - 'aya' = Ayachour's method to solve H\G iteratively
%
% inputs:
%   J:          Numeric matrix (possibly sparse) or function handle
%   b:          Vector, system rhs (b = -F(x) for the newton step)
%   type:       String, can be 'sim', 'giv' or 'aya'
%   epsilon_res:float, Residual tolerance.
%   mkryl:      maximum size of the krylov space
%   Pinv:       function handle, function computing v->inv(M)*v for
%               some preconditioning matrix M
% returns:
%   dx:         Vector, Solution of J\b (if solver converged)
%   j:          iter, Number of gmres iterations done
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

n = numel(b); % size of solution sought
m = mkryl;

H = zeros(m+1,m); % Hessenberg matrix (memory allocation)
switch type
  case 'giv'
    C = zeros(m+1,1); % Cosine component of Givens rotations
    S = zeros(m+1,1); % Sine component of Givens rotations
  case 'aya'
    u = zeros(m,1); % Residual vector.
    R = zeros(m,m); % Inverse Hessenberg sub-matrix (Hk).
end

% JFNK with preconditioning
% Krylov solution of J dx = b with preconditioning
V = zeros(n,m+1); % matrix with columns containing the Krylov vectors
beta = norm(b);
if beta == 0
  % RHS is zero
  dx = zeros(n,1);
  j = 0;
  return
end
V(:,1) = b/beta; % First vector of krylov space
% GMRES algorithm
G = beta*eye(m+1,1);
for j=1:m % Applied on all Krylov space dimension
  % One step in the Arnoldi algorithm
  [V,H,Hinv] = one_step_arnoldi(V,H,j,Pinv,J);

  % Preparatory steps for H\G solve
  switch type
    case 'sim' % solve system at each iteration
      yj  = H(1:j+1,1:j)\G(1:j+1); % solve for the minimum in the least square sense
      resj = norm(H(1:j+1,1:j)*yj - G(1:j+1));
    case 'giv' % Use Givens rotations.

      % Apply previous rotations to new H column
      H(1:j,j) = givens_rotation(C(1:j-1),S(1:j-1),H(1:j,j),j-1);
      % Calculate new rotations.
      normH = norm(H(j:j+1,j));
      if normH ~= 0
        C(j) = H(j,j)/normH;
        S(j) = -H(j+1,j)/normH;
        H(j:j+1,j) = [normH,0];
        % Apply new rotation to RHS (G)
        G(j:j+1) = givens_rotation(C(j),S(j),G(j:j+1),1);
      end

      resj = abs(G(j+1));

    case 'aya' % Ayachour method.
      % This follows the implementation of Theorem 3.1 of
      % A fast implementation for GMRES method, by E.H. Ayachour,
      % Journal of Computational and Applied Mathematics 159 (2003), 269 - 283

      % Build next part of R=inv(H_k') matrix, see
      % GMRES implementations and residual smoothing techniques for solving ill-posed linear systems,
      % by Matinfar, Zareamoghaddam, Eslami, and Saeidy,
      % Computers and Mathematics with Applications 63 (2012) 1-13
      % Algorithm 7, step 2.c.
      % R_j = (R_j-1 -R_j-1*g*Hinv)
      %       (         Hinv      )
      % From steps 2.c to 2.e, if H(j+1,j)=0, then the method
      % exits, or equivalently applies Hinv=1
      if j > 1
        R(1:j-1,j) = -R(1:j-1,1:j-1)*H(2:j,j)*Hinv;
      end
      R(j,j) = Hinv;
      u(j) = R(1:j,j)'*H(1,1:j)';
      c = 1 / sqrt(1 + u(1:j)'*u(1:j));
      resj = beta*c;
  end
  if (resj < epsilon_res) % stop increasing the krylov space size
    break
  end
end
% Compute solution
switch type
  case 'giv'
    yj = H(1:j,1:j)\G(1:j); % solve for the minimum in the least square sense
  case 'aya'
    yj = (beta*c*c)*R(1:j,1:j)*u(1:j);
end
% compute the delta x
dx = V(:,1:j)*yj;
if isa(Pinv,'function_handle')
  dx = Pinv(dx);
elseif ~isempty(Pinv)
  dx = Pinv*(dx);
end
end

function [V,H,Hinv] = one_step_arnoldi(V,H,j,Pinv,J)
% Perform a new step in the Arnoldi algorithm by approximating the Jacobian,
% computing a Gram-Schmidt step and filling the matrices V and H with the specified function handle.
% Hinv is an optional output and if required it is used to normalize the basis.
Hinv = 1;           % Default Hinv

% Right side preconditioner as in eq 28, D.A.Knoll  JCP 193 (2004) 357-397
dxstep = V(:,j);
if isa(Pinv,'function_handle')
  dxstep = Pinv(dxstep);
elseif ~isempty(Pinv)
  dxstep = Pinv*dxstep;
end
% Evaluate jacobian action
if isa(J,'function_handle')
  dFstep = J(dxstep);
else
  dFstep = J*dxstep;
end

% Modified Gram-Schmidt procedure for constructing Krylov vectors.
V(:,j+1) = dFstep; % Compute new basis
for i = 1:j
  H(i,j) = V(:,i)'*V(:,j+1); % Compute Hessenberg matrix with updated V(:,j+1)
  V(:,j+1) = V(:,j+1) - H(i,j)*V(:,i); % Arnoldi: subtract the component to make it orthogonal
end

H(j+1,j) = norm(V(:,j+1));
if H(j+1,j) ~= 0
  Hinv = 1 ./ H(j+1,j);
  V(:,j+1) = V(:,j+1) * Hinv; % normalize basis
end
end

function rotated = givens_rotation(cos_rot,sin_rot,mat_in,k)
% Applies Givens rotations on k columns to perform essentially a QR decomposition.
%
% These rotations are noted in the original GMRES paper:
%   GMRES: A generalized minimal residual algorithm for solving nonsymmetric linear systems
%   SIAM Journal on scientific and statistical computing, 1986
% but later papers refer to them as "Givens rotations".
rotated = mat_in;
for i = 1:k
  w1 = cos_rot(i)*rotated(i) - sin_rot(i)*rotated(i+1);
  w2 = sin_rot(i)*rotated(i) + cos_rot(i)*rotated(i+1);
  rotated(i:i+1) = [w1,w2];
end
end
