function [dx, j] = gmres_block(J, b, type, epsilon_res, mkryl, Pinv)
% GMRES_BLOCK Implements the Block-GMRES algorithm to solve J*dx = b
%
%   [dx, j] = gmres_block(J, b, type, epsilon_res, mkryl, Pinv)
%
% Reference for Block-GMRES:
% BLOCK KRYLOV SPACE METHODS FOR LINEAR SYSTEMS WITH MULTIPLE RIGHT-HAND
% SIDES: AN INTRODUCTION by MARTIN H. GUTKNECHT (2006)
%   https://people.math.ethz.ch/~mhg/pub/delhipap.pdf
%
% Implements three different methods:
%   - 'sim' = compute full QR decomposition of H at each iteration
%   - 'giv' = use Givens rotations to make H upper triangular
%   - 'qr'  = compute QR decomposition of H iteratively
%
% inputs:
%   J:          Numeric matrix (possibly sparse) or function handle
%   b:          Vector or matrix, system rhs (b = -F(x) for the newton step)
%   type:       variant for QR decomposition of H: sim/giv/qr
%   epsilon_res:float, Residual tolerance.
%   mkryl:      maximum size of the krylov space
%   Pinv:       right preconditioner for linear system
% returns:
%   dx:         Vector, Solution of J\b (if solver converged)
%   j:          iter, Number of gmres iterations done
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% Characteristic sizes
[n,s] = size(b);
m = mkryl;

% Hessenberg matrix
H = zeros((m+1)*s,m*s);
switch type
  case 'sim'
  case 'giv'
    C = zeros(m*s*s,1); % Cosine component of Givens rotations
    S = zeros(m*s*s,1); % Sine component of Givens rotations
    I = zeros(m*s*s,1); % Row index of Givens rotations
  case 'qr'
    Q = zeros(2*s,2*s*m); % Successive Q decompositions
end

% Krylov space
V = zeros(n,(m+1)*s);

[V(:,1:s),r0] = qr(b,0);

% RHS
G = eye((m+1)*s,s)*r0;

qj = G(1:s,:);
Nqj = sqrt(sum(qj(:).^2));

if Nqj < epsilon_res
  dx = zeros(n,s);
  j = 0;
  return
end

% Block-GMRES iterations
for j=1:m
  jbl = (j-1)*s+1:j*s; % j-th block
  
  % Block Arnoldi
  dx = V(:,jbl);
  % Apply preconditioner
  if isa(Pinv,'function_handle')
    dx = Pinv(dx);
  elseif ~isempty(Pinv)
    dx = Pinv*dx;
  end
  % Evaluate jacobian action
  if isa(J,'function_handle')
    yt = J(dx);
  else
    yt = J*dx;
  end
  % Compute Hessenberg matrix (MGS)
  for k=1:j
    kbl = (k-1)*s+1:k*s; % k-th block
    Vk = V(:,kbl);
    Hkj = Vk.'*yt;
    yt = yt-Vk*Hkj;
    H(kbl,jbl) = Hkj;
  end
  [V(:,jbl+s),H(jbl+s,jbl)] = qr(yt,0);

  % QR decomposition of H
  switch type
    case 'sim'
      [Qj,Rj] = qr(H(1:(j+1)*s,1:j*s));
      Gj = Qj(1:s,:).'*G(1:s,:); % Qj.' = inv(Qj)

      % Last block of RHS
      qj = Gj(jbl+s,:);
    case 'giv'
      Hj = H(1:(j+1)*s,jbl);
      % Apply previous rotations
      Hj = givens_rotation(C,S,I,Hj,(j-1)*s*s);
      % Rotations 1:(s,1) 2:(s-1,1) ... s:(1,1) s+1(s+1,2) ... 2*s:(2,2) ... s*s:(s,s)
      for k = 1:s*s
        ir = (j-1)*s*s+k;
        ik = 1+mod(k-1,s); % ik=1..s
        jk = 1+(k-ik)/s;   % jk=1..s
        ik = 1+s-ik;       % ik=s..1
        ik = ik+(jk-1);    % shift by number of columns
        ik = ik+(j-1)*s;   % add all previous blocks
        I(ir) = ik;
        normH = norm(Hj(ik:ik+1,jk));
        if normH ~= 0
          C(ir) =  Hj(ik  ,jk)/normH;
          S(ir) = -Hj(ik+1,jk)/normH;
          Hj(ik:ik+1,:) = givens_rotation(C(ir),S(ir),1,Hj(ik:ik+1,:),1);
          Hj(ik+1,jk)=0; % Enforce R format
        end
      end
      H(1:(j+1)*s,jbl) = Hj;
      % Apply new rotations to RHS (G)
      G = givens_rotation(C((j-1)*s*s+1:j*s*s),S((j-1)*s*s+1:j*s*s),I((j-1)*s*s+1:j*s*s),G,s*s);

      % Last block of RHS
      qj = G(jbl+s,:);
    case 'qr'
      Hj = H(1:(j+1)*s,jbl);
      % Apply previous Q
      for k=1:j-1
        Hj((k-1)*s+1:(k+1)*s,:) = Q(:,(k-1)*2*s+1:k*2*s).'*Hj((k-1)*s+1:(k+1)*s,:);
      end
      [Qj,Rj] = qr(Hj((j-1)*s+1:(j+1)*s,:));
      Q(:,(j-1)*2*s+1:j*2*s) = Qj;
      Hj((j-1)*s+1:(j+1)*s,:) = Rj;
      H(1:(j+1)*s,jbl) = Hj;

      % Apply new Q to RHS (G)
      G((j-1)*s+1:(j+1)*s,:) = Qj.'*G((j-1)*s+1:(j+1)*s,:);
      % Last block of RHS
      qj = G(jbl+s,:);
  end

  Nqj = sqrt(sum(qj(:).^2));

  if Nqj < epsilon_res, break; end
end

switch type
  case 'sim'
    % Use R and updated RHS
    kj = Rj(1:j*s,:)\Gj(1:j*s,:);
  case {'giv','qr'}
    % H is already upper triangular
    kj = H(1:j*s,1:j*s)\G(1:j*s,:);
end
Yj = V(:,1:j*s);
dx = Yj*kj;
% Apply preconditioner
if isa(Pinv,'function_handle')
  dx = Pinv(dx);
elseif ~isempty(Pinv)
  dx = Pinv*dx;
end
end

function R = givens_rotation(C,S,I,M,nr)
% Applies Givens rotations
%
% These rotations are noted in the original GMRES paper:
%   GMRES: A generalized minimal residual algorithm for solving nonsymmetric linear systems
%   SIAM Journal on scientific and statistical computing, 1986
% but later papers refer to them as "Givens rotations".
R = M;
for ir = 1:nr
  ik = I(ir);
  wk1 = C(ir)*R(ik,:) - S(ir)*R(ik+1,:);
  wk2 = S(ir)*R(ik,:) + C(ir)*R(ik+1,:);
  R(ik:ik+1,:) = [wk1;wk2];
end
end