function [hnormsq,nfeval,Fx,mask,Jdiag,J] = evaluate_F(F,x,opts,P,nfeval,optsJ)
% EVALUATE_F helper function for solvers
%
%   [hnormsq,nfeval,b,mask,Jdiag,J] = evaluate_F(F,x,opts,P,nfeval,optsJ)
%
% helper function taking F, evaluating it on x and computing much useful
% quantities like residual norms, and returning J function handle or J matrix
% if 6 outputs are expected
%
% inputs:
%   F:        Function handle, residual function, see solveF for more info
%   x:        Vector, point to evaluate F at
%   opts:     struct, parameter structure for the evaluation of F
%   P:        struct, parameter structure for the whole Newton solver
%   nfeval:   int, current number of evaluations of F
%   optsJ:    struct, parameter structure for the evaluation of J
% returns:
%   hnormsq   Half of the square of the norm of Fx: normsq=0.5*sum(Fx.^2)
%   nfeval:   int, updated number of evaluations of F
%   Fx:       Vector, the function evaluate at x
%   mask:     Vector of bool, mask indicating the rows where J has just a
%             diagonal element
%   Jdiag:    Vector, vector containing the diagonal elements of J, where
%             mask==true, otherwise zeros
%   J (opt.): Matrix, Sparse Matrix or Function handle, the jacobian
%             either defined as a function handle computing J(v) = Jac * v
%             for the jacobian-free methods or the full jacobian in sparse
%             or full format.
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% get F value, Jacobian if P.anajac=true and the mask
[Fx,~,J,~,~,rowmask] = F(x,opts);
nfeval = nfeval+1;

if nargout>=6
  switch P.algoF
    % if newton algorithm, the jacobian must be approximated if P.anajac=false
    case {'newton','broyden'}
      if ~P.anajac
        % of the matrix based solver should be used but an analytical jacobian is not
        % supplied by F, compute it using finite differencing scheme
        J = jacfd(F,{x,optsF},'F0',{Fx},'szF0',{numel(Fx)},'workers',P.workers);
        nfeval = nfeval+2*numel(x);
      end

    case 'jfnk'
      % building the J(v) function handle
      J = @(v_) directional_derivative(F,x,-Fx,v_,P,optsJ);

    otherwise
      error('Method algoF="%s" not supported.', P.algoF);
  end
end

% preparing mask
mask = logical(rowmask);
if ~P.userowmask, mask(:) = false; end % set the whole mask to false
Jdiag = rowmask;
hnormsq = 0.5*sum(Fx.^2);
end

function dFstep = directional_derivative(F,x,b,v,P,opts)
% dFstep = directional_derivative(F,x,Fx,v,P,opts)
% helper function, taking F and other quantities and returning the
% directional derivative of F in direction v using finite differencing
%
% inputs:
%   F:        Function handle, residual function, see solveF for more info
%   x:        Vector, point to evaluate directional derivative at
%   b:        Vector, rhs of newton iteration b=-F(x), could also be
%             computed here but its given to avoid calling F an unnecessary
%             second time
%   v:        Vector, direction to evaluate directional derivative in
%   P:        struct, parameter structure for the whole Newton solver
%   opts:     struct, parameter structure for the evaluation of F
% returns:
%   dFstep:   Vector, computed finite difference directional derivative

% Note: Additional function evaluations when a reduction of the step size
% is required is not tracked down anymore

nv = size(v,2);
dFstep = NaN(size(v));
for ii = 1:nv
  % Initial step to approximate the Jacobian
  epsilon_d = P.epsilon_d;
  v_ = epsilon_d*v(:,ii);
  dFstep_ = F(x+v_,opts) + b;

  % Reduce variational step
  while P.robust_on && any(~isfinite(dFstep_)) && epsilon_d > P.epsilon_d_tol
    epsilon_d = epsilon_d/10; % smaller step
    v_ = v_/10;
    dFstep_ = F(x+v_,opts) + b;
    if P.debug > 1
      fprintf(' *** Krylov space F(x + eps) = NaN try to reduce dx *** \n');
    end
  end
  if epsilon_d < P.epsilon_d_tol
    w = warning('off','backtrace');
    warning('Unable to compute directional derivative');
    warning(w);
    return
  end
  % divide by step size
  dFstep(:,ii) = dFstep_/epsilon_d;
end
end
