function [gamma,res,it] = rzpfastgr(alpha,c,dd,dmax,iter_max,tol)
%% [gamma,res,it] = rzpfastgr(alpha,c,dd,dmax,iter_max,tol)
% Function to be mexified. It comprises the fast (extension of the 
% Olfsson paper  K E J Olofsson 2022 Plasma Phys. Control. Fusion 64) 
% version for the calculation of the vetical growth rate (without cde only).
%
% Input: alpha:
%        c:
%        dd:
%        dmax:
%        iter_max:
%        tol:
% gamma: growth rate of the vertical instability (Hz)
% res: residual from the bisection algorithm
% it: numebr of iteratio of the bisection algorithm performed
%
% An example of the use of the calculation of the growth rate with
% different methods:
%
% [L,LX] = rzp(tok,shot,t);
%
% %% Using rzpfastgr
% [gamma,res,it] = rzpfastgr(LX.Ia,LX.Iu,LX.Iy,LX.Ip,LX.rIp,...
%        L.dd,L.N,L.F,L.dmax,L.RBrye,L.RBzye,L.drx,L.dzx,L.ny,L.np,L.ne,...
%        L.icde,L.P.griterm,L.P.grtol);
%
% %% Using rzpfastA
% A = rzpfastA(LX.Ia,LX.Iu,LX.Iy,LX.Ip,LX.rIp,LX.bp,...
%         L.Mey,L.Mee,L.Re,LX.Rp,L.RBrye,L.RBzye,L.drx,L.dzx,L.ny,L.np,L.ne,L.icde);
% gamma = max(real(eig(A)));
%
% %% Using fgess
% sys = fgess(L);
% A = sys.A;
% gamma = max(real(eig(A)));
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

%#codegen 
%Precalculate some quantities for the characteristic equation (for time saving)
c11_pre = c(:,1).*c(:,1);
c22_pre = c(:,2).*c(:,2);
c12_pre = c(:,1).*c(:,2);

%Store the characteristic equation
eig_determinant = @(lambda) solve_eq(dd,c11_pre,c22_pre,c12_pre,alpha,lambda);

%Calculate interval for the maximum positive eigenvalue
lambda_low = single(0);
lambda_upp = dmax + single(norm(c*alpha*c','fro'));

%Calculate the growth rate with the bisection algorithm
[lambda,res,it] = bisection(eig_determinant,lambda_low,lambda_upp,iter_max,tol);

gamma  = lambda;
end


function res = solve_eq(dd,c11_pre,c22_pre,c12_pre,alpha,lambda)
%% Characteric equation obrained from det( 1 + \alpha * c^T * (-\Lambda-\lambda)^{-1} * c ) = 0
% Solve the characteristic equation for the eigenvalues \lambda of the the 
% matrix A = \Lambda + c * \alpha * c^T.
% \Lambda is a ne diagonal matrix.
% c is a ne x 2 matrix. c = [c1 c2] with c1 and c2 column vectors of ne elements 
% \alpha is 2 x 2 matrix.
%
% dd: vector of diagonal elements of \Lambda
% c11_pre: precalculated vector from c1 .* c1
% c22_pre: precalculated vector from c2 .* c2 
% c12_pre: precalculated vector from c1 .* c2
% alpha: 2 x 2 matrix
% lambda: eigevalue which should set to 0 the characteristic equation when
% inserted 
%
% res: residual of the equation for a given lambda

  divisor = -1./(dd+lambda);
  c11 = divisor*c11_pre;
  c22 = divisor*c22_pre;
  c12 = divisor*c12_pre;
  a11 = alpha(1,1);
  a22 = alpha(2,2);
  a12 = alpha(1,2);
  a21 = alpha(2,1);
  
  res = 1 + a12*c12 +a21*c12 + a11*c11 + a22*c22 + a11*a22*c11*c22 + ...
      a12*a21*c12*c12 - a12*a21*c11*c22 - a11*a22*c12*c12;
end

function [lambda,res,it] = bisection(F,lambda_low,lambda_upp,iter_max,tol)
%% Bisection algorithm
% Find the zeros in F(\lambda) = 0 inside a given interval applying the
% bisection algorithm. There must be a unique zero inside the given
% interval, otherwise the given value could be any of the zeros.
%
% F: function to bisect
% lambda_low: lower limit of the interval
% lambda_upp: upper limit of the interval
% iter_max: maximum number of iteration
% tol: tolerance for breaking the algorithm

  sign_low = sign(F(lambda_low));
  sign_upp = sign(F(lambda_upp));

  lambda = (lambda_upp+lambda_low)./2;
  res = single(0);
  it = int32(0);
  
  if sign_low*sign_upp == 1
    lambda = single(0);
    return
  end
  
  for ii = 1:iter_max
    it = ii;
    res = F(lambda);
    
    if res == 0 || (lambda_upp-lambda_low)/2 < tol
      break
    end
    
    if sign(res) == sign_low
      lambda_low = lambda;
    else
      lambda_upp = lambda;
    end
    
    lambda = (lambda_upp+lambda_low)./2;
  end
end