function [res,Jx,Ju,Jxdot] = rzpFlin(L,LY)
%% [res,Jx,Ju,Jxdot] = rzpFlin(L,LY)
% Returns linearizations of rzp Force operator around equilibrium LY. 
%
% LY contains the equilibrium time slice and L is the associated parameter structure.
%
% The outputs are the linearisation matrices with respect to the states of the
% system: F = dFdIy*dIy + dFdag*dag + dFdIe*dIe + dFdCo*dCo
%
% The residual is F=[Fy;Fp;Fr;Fz]. 
%
% Fr and Fz are the residuals of the radial and vertical force balance as in
% M. Walker and D. Humphreys, Automatica, vol. 45, no. 3, pp. 665–674,
% 2009:
%
% Fr = \sum_i(2*pi*Ri*Ixi*Bzvac,i) + (mu0*Ip^2/2)*(ln(8*Rc/a*sqrt(k)) + betap + li/2 -3/2)
% Fz = -\sum_i(2*pi*Ri*Ixi*Brvac,i)
%
% where Brvac and Bzvac are the vacuum (only coil + vessel) radial and vertical
% magnetic field
%
% Fy is the expansion of Iy along with its parameters ag (Ip,Rc,Zc):
%
% Fy = dIy - (dIy/dRc)*dRc - (dIy/dZc)*dZc - (Iy0/Ip0)*dIp
% 
% where dIy/dRc = -dIy/dr and dIy/dZc = -dIy/dz are computed using finite difference method 
%
% The last equation Fp is:
%
% Fp = dIp - dIp'
%
% where dIp is the expansion parameter ag and dIp' the constraint Co. Physically,
% they are the same quantity, but this trick enables to make use of the more general 
% variables ag rather than Rc and Zc. The variables adopted are therefore
% the same for fge (Iy, ag, Ie and Co) and for any general forward model
% for the magnetic equilibrium calculation. 
%
% We introduce one more empty equation in case of cde to fit with the number 
% of equations in fgeFlin. In case of deployment of the cde from fgs to fge, 
% the addition row must be removed.
%
% The linearisation variables are Iy, ag = [Ip;Rc;Zc] (the expansion
% parameters for Iy), Ie = [Ia;Iu] and Co = [Ip;bp;qa/li] (the constraints, only li for the moment)
%
% Note that inserting the two trivial equations the numbers of equations
% and of the variables Iy and ag is the same.
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

assert(numel(LY.t)==1,'call rzpFlin for only a single time slice, use meqxk() to slice');
assert(all(contains(L.agconc(:,3),{'Ip','bp','li'})),'rzp:agcon','rzpl works only with constraints (Ip,bp,li)')
assert(L.ny==numel(LY.Iy),'L.ny (%d) does not match the numers of elements in LY.Iy (%d)',L.ny,numel(LY.Iy))

%% Compute fast version of the linearisation
% Compute only dFdag (split in dFydag and
% dFprzdag=[dFpdag;dFrdag;dFzdag]), dFprzdIe=[dFpdIe;dFrdIe;dFzdIe],
% dFpdCo.
% For more information, check rzpFlinfast
[dFydag,dFprzdag,~,dFprzdIe,dFprzdIp] = rzpFlinfast(L,LY);
         
%% Compute complete linearization w.r.t. Co
%for now, constraints are Ip if cde, betap and li
dFrdbp = mu0*(LY.Ip.^2)/2;
dFrdli = mu0*(LY.Ip.^2)/4;
dFprzdIpbpli = [dFprzdIp, [0, 0; dFrdbp, dFrdli; 0, 0]];

%% CDE
Rc = sum(L.rry(:).*LY.Iy(:))/LY.Ip;
Bz = L.Bzye*[LY.Ia;LY.Iu];
Gamma0 = -4*pi*L.rry(:)'.*LY.Iy(:)'*Bz(:)/(mu0*LY.Ip^2);
% definition from A. Coutlis, Nucl. Fusion, vol. 39, no. 5, pp. 663-683, 1999.
% Linked to the same paper definition of Gamma0 in rzpFlin
Lp = mu0*Rc*(Gamma0-0.5-LY.bp);

if all(L.icde)
  switch L.P.cde
    case 'OhmTor_rigid_0D'
      dLpdr = mu0*(Gamma0-0.5-LY.bp + 1); % as in FP thesis
      dLpdz = 0;
      dcdedIpdot = Lp; % linear cde wrt Ipdot
      dcdedRcdot = LY.Ip*dLpdr + 2*pi*sum(L.rry(:).*LY.Iy(:).*Bz(:))/(LY.Ip); %linear cde wrt Rcdot
      dcdedZcdot = LY.Ip*dLpdz; %linear cde wrt Zcdot (0)
      dcdedagdot = [dcdedIpdot dcdedRcdot dcdedZcdot];
      dcdedIedot = (L.Mey*LY.Iy(:))'/LY.Ip;

      dcdedIp  = LY.Rp;
      dcdedIni = -LY.Rp;
    otherwise
      error('rzp:cde','RZP linearisation not available for the cde selected');
  end
end

%% Order the force operator F according to L.ind and L.agconc
Jx = zeros(L.nN);
Ju = zeros(L.nN,L.nuN); % [Va;Co;IniD]
Jxdot = zeros(L.nrD,L.nN);
ixe = [L.ind.ixa,L.ind.ixu];
ire = ixe;

co2agDOF = eye(3);
if L.icde
  co2agDOF(1, :) = []; % removing the Fp residual if cde is used instead
end

% Rigid model equations
Jx(1:L.nN+1:L.nN*L.ny) = 1; % Only diagonal elements
Jx(L.ind.irGS,L.ind.ixg) = dFydag;
Jx(L.ind.irC ,L.ind.ixg) = co2agDOF * dFprzdag;
Jx(L.ind.irC ,      ixe) = co2agDOF * dFprzdIe;
Ju(L.ind.irC ,L.ind.iuC) = co2agDOF * dFprzdIpbpli * co2agDOF.';

% CDE
if L.icde
  Jx(L.ind.irp,L.ind.ixg(1)) = dcdedIp; % Ip is first ag state
  Ju(L.ind.irp,L.ind.iuni) = dcdedIni;
  irDp = L.ind.irp - (L.nN-L.nrD);
  Jxdot(irDp,L.ind.ixg) = dcdedagdot;
  Jxdot(irDp,      ixe) = dcdedIedot;
end

% Circuit equations
Jx(      ire,      ixe) = diag(L.Re);
Ju(L.ind.ira,L.ind.iua) = -eye(L.G.na); % dFdVa
irDe = ire - (L.nN-L.nrD);
Jxdot(irDe,      ixe ) = L.Mee;
Jxdot(irDe,L.ind.ixGS) = L.Mey;

% rescaling
Jx    = L.resscal.*Jx   .*L.xscal.';
Ju    = L.resscal.*Ju              ;
Jxdot = L.resscal(L.ind.irD).*Jxdot.*L.xscal.';

% Residual (assume zero)
res = zeros(L.nN,1);

end
