function [aW, FW] = rtciwall(L, LY)
% RTCIWALL Computes wall gaps from equilibrium in LY
% [aW, FW] = rtciwall(L, LY)
%
% Computes wall gaps of the plasma. Internally utilizes find_contours.m to
% find LCFS.
% Arguments:
%   L:    struct, standard L structure
%   LY:   struct, standard LY structure containing converged GS-equilibrium
% returns:
%   aW:   double(nW,nFW), wall gaps
%   FW:   double( 1,nFW), flux values at line-of-sight-plasma intersections
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

aW = LY.aW; % aW init
if isempty(aW)
  aW  = zeros(L.G.nW,L.P.nFW);
end

% wall gap flux values are computed from LCFS flux values
if LY.nB
  FW = meqFW(LY.FB(LY.nB),LY.FX,LY.lX(LY.nB),LY.nX,L.P.nFW);
else
  FW = repmat(L.FN,1,L.P.nFW);
end

% only compute wall-gaps if nW > 0
if L.nW
  crW = L.crW; czW = L.czW; rO = L.G.rW; zO = L.G.zW;
  FoW = bintmex(LY.Fx,int32(L.kxW-1),L.cWx); % flux values at wall origin points
  [~, ~, aW] = find_contours(L, LY, FW, aW, rO, zO, crW, czW, FoW, int8(0));
  % clamping to min/max
  aW = min(max(aW,0),L.G.aW);
end
end