function [rq, zq, aq, rO, zO, crq, czq, nDeff] = rtciplasma(L, LY)
% RTCIPLASMA Computes flux surfaces from equilibrium in LY
% [rq, zq, aq, rO, zO, crq, czq] = rtciplasma(L, LY)
%
% Computes flux surface locations inside the plasma. Internally utilizes
% find_contours.m and implements special a special case for doublets with
% mantle
% Arguments:
%   L:    struct, standard L structure
%   LY:   struct, standard LY structure containing converged GS-equilibrium
% returns:
%   rq:   double(noq,npq,nD), r-positions of discretized flux surfaces
%   zq:   double(noq,npq,nD), z-positions of discretized flux surfaces
%   aq:   double(noq,npq,nD), radial positions (how far away from origin 
%                               points) for the discretized flux surfaces
%   rO:   double(noq,nD), r-positions of constant-theta ray origins
%   zO:   double(noq,nD), z-positions of constant-theta ray origins
%   crq:  double(noq,nD), r-directions of constant-theta rays
%   czq:  double(noq,nD), z-directions of constant-theta rays
%   nDq:  int32, number of domains for which flux surfaces are computed, nDq <= nD
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

nAeff = min([LY.nA,LY.nB,L.nD]);
is_doublet = L.nD == 3 && LY.nA == 2 && LY.nB == 3;
nDeff = nAeff;

% Init
rq = zeros(L.noq, L.npq, L.nD); % positions of points on flux surfaces
zq = rq;
rO = zeros(L.noq, L.nD); % origin points
zO = rO;
crq = rO; % constant-theta ray directions
czq = rO;

aq = LY.aq;
if isempty(aq) % aq init
  aq = rq;
  for iD = 1:nAeff
    bbox_size = min([LY.rA(iD)-L.G.rx(1),L.G.rx(end)-LY.rA(iD),LY.zA(iD)-L.G.zx(1),L.G.zx(end)-LY.zA(iD)]);
    aq(:,:,iD) = repmat(bbox_size*L.pinit*L.pq,L.noq,1);
  end
end

FQ = (LY.F1 - L.fq .* (LY.F1 - LY.F0)); % psi values of flux surfaces to track
for iD=1:nAeff % all axis domains
  [rq(:, :, iD), zq(:, :, iD), aq(:, :, iD)] = ...
    find_contours(L, LY, FQ(iD, :), aq(:, :, iD), LY.rA(iD), LY.zA(iD), L.crq, L.czq, LY.FA(iD), int8(iD));
  % for domains with axes, the axis is always the origin point
  rO(:, iD) = LY.rA(iD); zO(:, iD) = LY.zA(iD);
  % for domains with axes, the ray directions are predefined
  crq(:, iD) = L.crq; czq(:, iD) = L.czq;
end

if is_doublet % doublet special case
  % a special version of the flux surface finding algorithm has to be
  % called for the mantle domain which does not have an axis
  [rq(:, :, 3), zq(:, :, 3), aq(:, :, 3), rO(:, 3), zO(:, 3), crq(:, 3), czq(:, 3)] = ...
    mantle_domain_contours(L, LY, rq, zq, FQ(3, :));
  nDeff = 3;
end
end


function [rm, zm, am, rO, zO, cr, cz] = mantle_domain_contours(L, LY, rq, zq, F)
% MANTLE_DOMAIN_CONTOURS Computes flux surfaces for a mantle domain in a doublet
% [rq, zq, aq, rO, zO, crq, czq] = mantle_domain_contours(L, LY, rq, zq, F)
%
% For the mantle, constant-theta-ray directions have to be adapted to the
% geometry and are always different and thus a return. For the mantle, the
% origin points for the constant-theta rays form the separatrix
% Arguments:
%   mode: string, 'plasma' or 'wall', flag indicating if flux surfaces or
%           wall-gaps should be computed
%   L:    struct, standard L structure
%   LY:   struct, standard LY structure containing converged GS-equilibrium
% returns:
%   rm:   double(noq,npq,nD), r-positions of discretized flux surfaces
%   zm:   double(noq,npq,nD), z-positions of discretized flux surfaces
%   am:   double(noq,npq,nD), radial positions (how far away from origin 
%                               points) for the discretized flux surfaces
%   rO:   double(noq,nD), r-positions of separatrix points
%   zO:   double(noq,nD), z-positions of separatrix points
%   cr:   double(noq,nD), r-directions of constant-theta rays
%   cz:   double(noq,nD), z-directions of constant-theta rays
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% separatrix r and z positions are taken from the LCFS of each lobe
rS = zeros(L.noq, 2); zS = zeros(L.noq, 2); % separatrix points
for iD=1:2 % for the two axis domains
  % for each lobe, the points have to be ordered according to their angle
  % w.r.t. the X-point such that they can be contiguously combined from the two lobes
  [rS(:, iD), zS(:, iD)] = x_point_ordering(rq(:, end, iD), zq(:, end, iD), ...
                                            LY.rX(1), LY.zX(1), LY.rA(iD), LY.zA(iD));
end
rO = rS(:); rO = rO(1:2:end); % concatenating and subsampling to get back to L.noq points
zO = zS(:); zO = zO(1:2:end);

% compute good directions for the rays to have good sampling around X-point
[cr, cz] = get_mantle_directions(rO, zO, LY.rX(1), LY.zX(1), LY.rA, LY.zA);

am = zeros(L.noq, L.npq);
FO = repmat(LY.FB(1), L.noq, 1);
[rm, zm, am] = find_contours(L, LY, F, am, rO, zO, cr, cz, FO, int8(3));
end


function [cr, cz] = get_mantle_directions(rO, zO, rX, zX, rA, zA)
% GET_MANTLE_DIRECTIONS computes constant-theta ray directions for a doublet
% [cr, cz] = get_mantle_directions(ro, zo, rX, zX, rA, zA)
%
% For the top of the top lobe and the bottom of the bottom lobe, ray
% directions are still originating from the axis. For the region around the
% X-point, ray directions go towards a fictitious point to the right (P_RHS)
% and the left of the doublet (P_LHS).
%             _____
%           /       \
%          /         \
%          |    A1   |
%           \       /
%             \   /
%   P_LHS       X       P_RHS
%             /   \
%           /       \
%          |    A2   |
%          \         /
%           \_______/
%           
% Arguments:
%   rO: double(noq,nD), r-positions of separatrix points
%   zO: double(noq,nD), z-positions of separatrix points
%   rX: double, r-position of central X point
%   zX: double, z-position of central X point
%   rA: double(2), r-positions of two axes
%   zA: double(2), z-positions of two axes
% returns: (mode='plasma')
%   cr:  double(noq,nD), r-directions of constant-theta rays
%   cz:  double(noq,nD), z-directions of constant-theta rays
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% RHS and LHS points are on a plane orthogonal to the A1-A2 vector centered
% at the X point and as far out as the distance between A1 and A2
rRHS = rX + (zA(1) - zA(2)); zRHS = zX - (rA(1) - rA(2));
rLHS = rX - (zA(1) - zA(2)); zLHS = zX + (rA(1) - rA(2));

% mask if a point on the separatrix should use the upper lobe axis as
% origin or the RHS or LHS points (the lower lobe axis as origin is used if
% all of these are not true)
maskA1 = (rO - rX) * (rA(1) - rA(2)) + (zO - zX) * (zA(1) - zA(2)) > 0; % upper lobe
maskRHS = inpolygon(rO, zO, [rA(1);rX;rA(2);rRHS], [zA(1);zX;zA(2);zRHS]); % right side of doublet
maskLHS = inpolygon(rO, zO, [rA(1);rX;rA(2);rLHS], [zA(1);zX;zA(2);zLHS]); % left side of doublet

% add axis origin rays according to mask
cr = (rO - rA(1)) .* maskA1 + (rO - rA(2)) .* ~maskA1;
cz = (zO - zA(1)) .* maskA1 + (zO - zA(2)) .* ~maskA1;

% add RHS and LHS origin rays acoording to mask
cr(maskRHS) = rRHS - rO(maskRHS); cz(maskRHS) = zRHS - zO(maskRHS);
cr(maskLHS) = rLHS - rO(maskLHS); cz(maskLHS) = zLHS - zO(maskLHS);

% norm ray directions
c_norm = sqrt(cr.*cr + cz.*cz);
cr = cr ./ c_norm; cz = cz ./ c_norm;
end


function [rq_ordered, zq_ordered] = x_point_ordering(rq, zq, rX, zX, rA, zA)
% X_PONT_ORDERING helper function for ordering flux surface points by angle
%   relative to the X-point angle
% [rq_ordered, zq_ordered] = x_point_ordering(rq, zq, rX, zX, rA, zA)
%
% Orders rq, zq by angle(X-point, Axis, (rq,zq))
% Arguments:
%   rq: double(k), r-positions of flux surface points, flattened
%   zq: double(k), z-positions of flux surface points, flattened
%   rX: double, r-position of X point
%   zX: double, z-position of X point
%   rA: double, r-position of axis
%   zA: double, z-position of axis
% returns:
%   LY:   struct, input LY structure equipped with rS and zS
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

angle = atan2(rq - rA, zq - zA);
angleX = atan2(rX - rA, zX - zA);
% have to add epsilon here for the case the X-point itself is part of rq,zq
[~, order] = sort(mod(angle - angleX + sqrt(eps), 2 * pi));
rq_ordered = rq(order); zq_ordered = zq(order);
end
