function [a,s] = rtcics(a,Fx,F,Opy,ro,zo,Opo,cr,cz,L)
% RTCICS Flux-surface contouring with Psi-toolbox method
%   [a,s] = rtcics(a,Fx,F,Opy,ro,zo,Opo,crq,czq,L)
% a is the distance of the contours Fx=F from the origin points ro,zo
% s is true if the method converged and false otherwise
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

rk = L.taur;
zk = L.tauz;

% spline coefficients
c = L.Mr*(L.Mz*Fx).';

niter = L.P.iterq;
tol = L.P.tolq*L.drx;

% alloc
mo = numel(ro);
np = numel(F);
no = numel(cr);
nz = L.nzx;
nr = L.nrx;

% fuq
sith = repmat(cz        ,1,np);
coth = repmat(cr        ,1,np);
s2th = repmat(cz.^2     ,1,np);
c2th = repmat(cr.^2     ,1,np);
scth = repmat((cz.*cr*2),1,np);
psth = repmat(F(:).'    ,no,1);
if mo>1 && np>1
  ro = repmat(ro,1,np);
  zo = repmat(zo,1,np);
end

% initial guess, avoid 0
if isempty(a)
  a = repmat(tol,no,np);
end

% Value at origin points
Fo = cs2devalmex(rk,zk,c,ro(:),zo(:),0);

% Gauss-Newton iteration
s = false;
for kiter = 1:niter
  r = ro(:) + a(:) .* coth(:);
  z = zo(:) + a(:) .* sith(:);
  [f,gr,gz,hrr,hrz,hzz] = cs2devalmex(rk,zk,c,r,z,[0,1,2]);
  p   = f;
  dp  = gz  .* sith(:) + gr  .* coth(:);
  d2p = hrr .* c2th(:) + hrz .* scth(:) + hzz .* s2th(:);
  da = (sqrt(max(dp.^2+2*d2p.*(psth(:)-p),0)).*sign(dp) - dp) ./ d2p;
  % Proximity to domain boundary
  jr = min(floor(max((r-L.G.rx(1))*L.idrx,0)),nr-2);
  jz = min(floor(max((z-L.G.zx(1))*L.idzx,0)),nz-2);
  kq = (nz-2)*(jr-1) + jz;
  Op0 = zeros(no,np,'int8'); Op1 = Op0; Op2 = Op0; Op3 = Op0;
  mask = (jr>0    & jz>0   ); Op0(mask) = Opy(kq(mask)     ); % lower left
  mask = (jr<nr-2 & jz>0   ); Op1(mask) = Opy(kq(mask)+nz-2); % lower right
  mask = (jr>0    & jz<nz-2); Op2(mask) = Opy(kq(mask)   +1); % upper left
  mask = (jr<nr-2 & jz<nz-2); Op3(mask) = Opy(kq(mask)+nz-1); % upper right
  zz = cat(3,Op0,Op1,Op2,Op3) - Opo;
  % Classify point with:
  %  ptype=0 means all neighbours are in the same domain as origin
  %  ptype=1 means some neighbours are in another domain
  %  ptype=2 means all neighbours are in another domain
  ptype = zeros(no*np,1);
  ptype(Opo & any(zz,3)) = 1; % Adding Opo disables domain checks for gaps
  ptype(Opo & all(zz,3)) = 2;
  % Wrong domain (if in plasma domain)
  mask = (ptype == 2);
  if any(mask)
    da(mask) = -L.drx;
  end
  % Derivative reverses close to boundary (wrong branch of X-point)
  mask = dp.*(Fo - psth(:)) > 0 & ptype == 1;
  if any(mask)
    da(mask) = -L.drx;
  end
  % Escape regions of incorrect gradient
  mask = dp.*(Fo - psth(:)) > 0 & ptype == 0;
  if any(mask)
    da(mask) = sign((p(mask)-psth(mask)).*dp(mask))*L.drx;
  end
  %
  a(:) = max(a(:) + da,0);
  % Check convergence
  if max(abs(da)) < tol, s = true; break, end
end
