function titstr = meqtitle(L,LY,quantities)
% MEQTITLE automated title string generation for meq plots
%
% title_string = meqtitle(L,LY,quantities);
% 
% L, LY: MEQ structures, where LY is a single time slice
% quantities: a cell array with strings corresponding to LY fields to be
% displayed. Default: {'shot','t','Ip'}
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if numel(LY.t)>1
  error('only single time slice supported for meqtitle, use meqxk() to slice');
end

if nargin==2 || isempty(quantities)
  quantities = {'shot','t','Ip'};
end

mystr = cell(numel(quantities),1);
for ii=1:numel(quantities)
  myquantity = quantities{ii};

  text=''; format = '%.1f'; pre=true; % default
  switch myquantity
    case 'Ip'
      unit = 'A';
    case 'shot'
      unit = '';  format='%d'; pre = false; text = '#';
    case 't'
      unit = 's'; format='%g';
    otherwise
      error('%s not implemented',myquantity)
  end
  
  val = LY.(myquantity);
  scal10 = floor(log10(val));
  if ~pre
    preunit = ''; fac = 1;
  elseif scal10 >= 6
    preunit = 'M'; fac = 1e-6;
  elseif scal10 >= 3
    preunit = 'k'; fac = 1e-3; 
  elseif scal10>= 0
    preunit = '';  fac = 1;
  else
    preunit = 'm'; fac = 1e3;
  end
  unit_ = [preunit,unit];

  formatstr = [' %s',format,'%s'];
  mystr{ii} = sprintf(formatstr,text,fac*val,unit_);
end

titstr = strtrim(strcat(mystr{:}));

end