%GSZRMEX  Solve Poisson equation
%
% This is the MATLAB equivalent implementation of libmeq/gszr.c.
% A more detailed help is available in GSZRMEX.
%
% If Fb and Iy or 3D arrays with matching sizes of the last dimension then
% it solves all problems corresponding to each page of Fb and Iy
% concurrently by vectorizing all operations on the FX and P arrays.
%
% See also GSZRMEX
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
function Fx = gszrmexm(Fb,Iy,cx,cq,cr,cs,ci,co,dz)

[nz2,nr2,ntot] = size(Iy);

nr1 = nr2+1;
nz1 = nz2+1;
nr  = nr1+1;
nz  = nz1+1;

nb = 2*(nz+nr2);

if (~isequal(numel(Fb),nb*ntot)), error('gszr:size','Fb should be of size %s but is actually of size %s',mat2str([nb,1,ntot]),mat2str(sz)); end
Fb = reshape(Fb,nb,1,ntot);

p  = zeros(nz2,nr2,ntot);
Fx = zeros(nz ,nr ,ntot);

i = 2:nz1;
j = 2:nr1;
Fx(i,j,:) = repmat(cx',nz2,1,ntot).*Iy;

i = 1:nz;
Fx(i,1 ,:)  = Fb(i,1,:);
Fx(i,nr,:)  = Fb(i+nz+2*nr2,1,:);
Fx(i,2 ,:)  = Fx(i,2  ,:) + ci*Fb(i,1,:);
Fx(i,nr1,:) = Fx(i,nr1,:) + co*Fb(i+nz+2*nr2,1,:);
Fx( 1,j,:) = Fb(nz+(1:2:2*nr2),1,:);
Fx(nz,j,:) = Fb(nz+(2:2:2*nr2),1,:);

l = nz1/2;
i = 3:2:nz1;
ip = i-1;
p(ip,:,:) = 2.*Fx(i,j,:);
Fx(i,j,:) = Fx(i+1,j,:)+Fx(i-1,j,:);
p = gsp(p,cq,cr,cs,nr2,l,ip);
Fx = gsu(Fx,p,nr1,i,ip);

lo = nz1/4; li = lo*2; id = 2; ih = 1; io = 4;
while lo > 1
  ii = id+ih;
  i = io+1:io:nz1;
  ip = i-1;
  p(ip,:,:) = Fx(i,j,:)-Fx(i+ii,j,:)-Fx(i-ii,j,:);
  Fx(i,j,:) = Fx(i,j,:)-Fx(i+ih,j,:)-Fx(i-ih,j,:)+Fx(i+id,j,:)+Fx(i-id,j,:);
  p(ip,:,:) = p(ip,:,:)+Fx(i,j,:);
  for l = lo:li:nz1
    p = gsp(p,cq,cr,cs,nr2,l,ip);
  end
  Fx = gsu(Fx,p,nr1,i,ip);
  id = id*2; ih = ih*2; io = io*2; lo = lo/2; li = li/2;
end

ii = io; io = id;
while ih >= 1
  i = io+1:ii:nz1;
  ip = i-1;
  p(ip,:,:) = 2*Fx(i,j,:)+Fx(i+id,j,:)+Fx(i-id,j,:);
  Fx(i,j,:) = Fx(i,j,:)-Fx(i+ih,j,:)-Fx(i-ih,j,:);
  for l = lo:li:nz1
    p = gsp(p,cq,cr,cs,nr2,l,ip);
  end
  Fx = gsu(Fx,p,nr1,i,ip);
  id = id/2; ih = ih/2; io = io/2; ii = ii/2; lo = lo*2; li = li*2;
end

i = 2:2:nz1;
ip = i-1;
p(ip,:,:) = 2.*Fx(i,j,:)+Fx(i+1,j,:)+Fx(i-1,j,:);
Fx(i,j,:) = 0.;
p = gsp(p,cq,cr,cs,nr2,lo,ip);
Fx = gsu(Fx,p,nr1,i,ip);

% Shift solution
Fx = Fx + (Fx([2:end end],:,:)-Fx([1 1:end-1],:,:)).*[dz;repmat(0.5*dz,nz2,1);dz];
end

function p = gsp(p,cq,cr,cs,nr2,l,ip)
for jj = nr2-1:-1:1
  p(ip,jj,:) = p(ip,jj,:) + cr(jj,l)*p(ip,jj+1,:);
end
p(ip,:,:) = cq(:,l).'.*p(ip,:,:);
for jj = 2:nr2
  p(ip,jj,:) = p(ip,jj,:) + cs(jj,l)*p(ip,jj-1,:);
end
end

function Fx = gsu(Fx,p,nr1,i,ip)
jj = 2:nr1;
Fx(i,jj,:) = Fx(i,jj,:) + p(ip,:,:);
end