%BFEF  Constant and polynomial base functions for p',TT'
% BFEF uses P=[nP nT] and produces base functions for p',TT' as the first
% nP<=4,nT<=4 functions in {1, F-F1, (F-F1)*(F-F0),
% (F-F1)*(F-F0)*(2*F-F1-F0)}.
%
% For details see: [MEQ-redbook]
%
% See also BFCT BFHELP.
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

function [varargout] = bfef(mode,P,varargin)

% Handle parameters
nP = P(1);
nT = P(2);
ng = nP + nT;
fPg = [ ones(nP,1) ; zeros(nT,1)];
fTg = [zeros(nP,1) ;  ones(nT,1)];

% Prepare structure for bfct
Pbfct.name = 'bfef';
Pbfct.fPg  = fPg;
Pbfct.fTg  = fTg;
Pbfct.f    = [];
Pbfct.fN   = @(FQ) bfef_fN(P,FQ);
Pbfct.fag  = @(FBA) bfef_fag(P,FBA);
Pbfct.fA   = [];
Pbfct.df   = [];
Pbfct.dfN  = @(FQ) bfef_dfN(P,FQ);
Pbfct.dfag = @(FBA) bfef_dfag(P,FBA);
Pbfct.dfA  = [];

switch mode
  case 6
    % Regularization constraints
    % [QQG,XQ          ] = BF(6,PAR, ~,F0,F1,R0,IR0,IDS)

    % Handle inputs
    [~,FA,FB,rA,irA,ids] = deal(varargin{:});

    % Total number of constraints
    nPq = max(nP-2,0);
    nTq = max(nT-2,0);
    nq = nPq + nTq + 2*(ng>0);
    % Regularization on second derivative of each profile
    FBA = FB-FA;
    Qqg = zeros(2,4);
    Qqg(1,3) =       2 *FBA^2*ids;
    Qqg(2,4) = sqrt(12)*FBA^3*ids;
    % Dispatch to 1...ng
    y1 = zeros(nq,ng); % Qqg
    y2 = zeros(nq,1);  % Xq
    y1(     1:nPq ,    1:nP ) = Qqg(1:nPq,1:nP)* rA;
    y1(nPq+(1:nTq),nP+(1:nT)) = Qqg(1:nTq,1:nT)*irA;
    % Adding constraint to minimise jphi at the boundary, while minimising
    % the difference between the p',TT' contributions therein.
    if nP > 0, y1(nq+(-1:0),   1) =  rA*ids*[ 1; 1]; end
    if nT > 0, y1(nq+(-1:0),nP+1) = irA*ids*[ 1;-1]; end

    % Assign outputs
    varargout = {y1,y2};
  case 16
    % Derivatives of regularization constraints
    % [DQGF0,DQGF1,DQGR0,...] = BF(16,PAR, ~,F0,F1,R0,IR0,IDS)

    % Handle inputs
    [~,FA,FB,rA,irA,ids] = deal(varargin{:});

    % Total number of constraints
    nPq = max(nP-2,0);
    nTq = max(nT-2,0);
    nq = nPq + nTq + 2*(ng>0);
    % Regularization on second derivative of each profile
    FBA = FB-FA;
    Qqg = zeros(2,4);
    Qqg(1,3) =       2 *FBA^2*ids;
    Qqg(2,4) = sqrt(12)*FBA^3*ids;
    dQqgdFB = zeros(2,4);
    dQqgdFB(1,3) =         4 *FBA  *ids;
    dQqgdFB(2,4) = 3*sqrt(12)*FBA^2*ids;
    % Dispatch to 1...ng
    y1 = zeros(nq,ng); % dQqgdFA
    y1(     1:nPq ,    1:nP ) = -dQqgdFB(1:nPq,1:nP)* rA;
    y1(nPq+(1:nTq),nP+(1:nT)) = -dQqgdFB(1:nTq,1:nT)*irA;
    y2 = -y1;          % dQqgdFB
    y3 = zeros(nq,ng); % dQqgdrA
    y3(     1:nPq ,    1:nP ) = Qqg(1:nPq,1:nP);
    y4 = zeros(nq,ng); % dQqgdirA
    y4(nPq+(1:nTq),nP+(1:nT)) = Qqg(1:nTq,1:nT);
    % Constraint for jphi at the boundary
    if nP > 0, y3(nq+(-1:0),   1) = ids*[ 1; 1]; end
    if nT > 0, y4(nq+(-1:0),nP+1) = ids*[ 1;-1]; end

    % Assign outputs
    varargout = {y1,y2,y3,y4};
  otherwise
    % Call generic bfct
    [varargout{1:nargout}] = bfct(mode,Pbfct,varargin{:});
end
end

function [gQ,IgQ] = bfef_fN(P, FQ)
% BFEF_FN Compute BFAB normalized basis functions

FQ = FQ(:);
nF = numel(FQ);
% Compute normalized basis functions
 gQ = [ones(nF,1),...
       (FQ-1),...
       (FQ-1).*FQ,...
       (FQ-1).*FQ.*(2*FQ - 1)];
% Compute normalized basis functions primitives
IgQ = [(FQ-1),...
       (FQ-1).^2/2,...
       (FQ-1).^2/6.*(FQ*2 + 1),...
       (FQ-1).^2/2.*(FQ).^2];
% Dispatch to 1...ng
 gQ =  gQ(:,[1:P(1),1:P(2)]);
IgQ = IgQ(:,[1:P(1),1:P(2)]);
end

function [alphapg,alphag] = bfef_fag(P, FBA)
% BFEF_FAG Conversion factors from normalized to physical basis functions

alphapg = FBA.^([1:P(1),1:P(2)].'-1);
alphag  = FBA.*alphapg;
end

function [dgQ,dIgQ] = bfef_dfN(P, FQ)
% BFEF_DFN Compute derivatives of BFEF normalized basis functions

FQ = FQ(:);
% Compute normalized basis functions derivatives
 dgQ = [zeros(size(FQ)),...
        ones(size(FQ)),...
        2*FQ-1,...
        (2*FQ - 1).^2 + 2*(FQ-1).*FQ];
% Compute derivatives of normalized basis functions primitives
dIgQ = [ones(size(FQ)),...
        (FQ-1),...
        (FQ-1).*FQ,...
        (FQ-1).*FQ.*(2*FQ - 1)];
% Dispatch to 1...ng
 dgQ =  dgQ(:,[1:P(1),1:P(2)]);
dIgQ = dIgQ(:,[1:P(1),1:P(2)]);
end

function [dalphapg,dalphag] = bfef_dfag(P, FBA)
% BFEF_DFAG Derivatives of conversion factors from normalized to physical basis functions

u = ([1:P(1),1:P(2)].'-1);
dalphapg =  u   .*FBA.^(u-1);dalphapg(u==0) = 0;
dalphag  = (u+1).*FBA.^ u   ;
end
