%BFAB  Polynomial base functions for p',TT'
% BFAB uses P=[nP nT] and produces base functions for p',TT' as the first
% nP<=3,nT<=3 functions in {F-F1, (F-F1)*(F-F0),
% (F-F1)*(F-F0)*(2*F-F1-F0)}.
%
% For details see: [MEQ-redbook]
%
% See also BFCT BFHELP.
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

function [varargout] = bfab(mode,P,varargin)

% Handle parameters
nP = P(1);
nT = P(2);
ng = nP + nT;
fPg = [ ones(nP,1) ; zeros(nT,1)];
fTg = [zeros(nP,1) ;  ones(nT,1)];

% Prepare structure for bfct
Pbfct.name = 'bfab';
Pbfct.fPg  = fPg;
Pbfct.fTg  = fTg;
Pbfct.f    = [];
Pbfct.fN   = @(FQ) bfab_fN(P,FQ);
Pbfct.fag  = @(FBA) bfab_fag(P,FBA);
Pbfct.fA   = [];
Pbfct.df   = [];
Pbfct.dfN  = @(FQ) bfab_dfN(P,FQ);
Pbfct.dfag = @(FBA) bfab_dfag(P,FBA);
Pbfct.dfA  = [];

switch mode
  case 6
    % Regularization constraints
    % [QQG,XQ          ] = BF(6,PAR, ~,F0,F1,R0,IR0,IDS)

    % Handle inputs
    [~,FA,FB,rA,irA,ids] = deal(varargin{:});

    % Total number of constraints
    nPq = max(nP-1,0);
    nTq = max(nT-1,0);
    nq = nPq + nTq;
    % Regularization on second derivative of each profile
    FBA = FB - FA;
    Qqg = zeros(2,4);
    Qqg(1,2) =       2 *FBA^2*ids;
    Qqg(2,3) = sqrt(12)*FBA^3*ids;
    % Dispatch to 1...ng
    y1 = zeros(nq,ng); % Qqg
    y2 = zeros(nq,1);  % Xq
    y1(     1:nPq ,    1:nP ) = Qqg(1:nPq,1:nP)* rA;
    y1(nPq+(1:nTq),nP+(1:nT)) = Qqg(1:nTq,1:nT)*irA;

    % Assign outputs
    varargout = {y1,y2};
  case 7
    % Inequality constraints
    % [QCG,XC          ] = BF(7,PAR, ~,F0,F1           )

    % Handle inputs
    [~,FA,FB] = deal(varargin{:});

    % number of constraints
    nc = min(nP,2);
    % Constraint for positive pressure
    FBA = FB - FA;
    Qcg = [3.0,FBA,0;...
           1.0,FBA,FBA*FBA];
    % Dispatch to 1...ng
    y1 = zeros(nc,ng); % Qcg
    y2 = zeros(nc,1);  % Xc
    y1(1:nc,1:nP) = Qcg(1:nc,1:nP);

    % Assign outputs
    varargout = {y1,y2};
  case 16
    % Derivatives of regularization constraints
    % [DQGF0,DQGF1,DQGR0,...] = BF(16,PAR, ~,F0,F1,R0,IR0,IDS)

    % Handle inputs
    [~,FA,FB,rA,irA,ids] = deal(varargin{:});

    % Total number of constraints
    nPq = max(nP-1,0);
    nTq = max(nT-1,0);
    nq = nPq + nTq;
    % Regularization on second derivative of each profile
    FBA = FB - FA;
    Qqg = zeros(2,4);
    Qqg(1,2) =       2 *FBA^2*ids;
    Qqg(2,3) = sqrt(12)*FBA^3*ids;
    dQqgdFB = zeros(2,4);
    dQqgdFB(1,2) =         4 *FBA  *ids;
    dQqgdFB(2,3) = 3*sqrt(12)*FBA^2*ids;
    % Dispatch to 1...ng
    y1 = zeros(nq,ng); % dQqgdFA
    y1(     1:nPq ,    1:nP ) = -dQqgdFB(1:nPq,1:nP)* rA;
    y1(nPq+(1:nTq),nP+(1:nT)) = -dQqgdFB(1:nTq,1:nT)*irA;
    y2 = -y1;          % dQqgdFB
    y3 = zeros(nq,ng); % dQqgdrA
    y3(     1:nPq ,    1:nP ) = Qqg(1:nPq,1:nP);
    y4 = zeros(nq,ng); % dQqgdirA
    y4(nPq+(1:nTq),nP+(1:nT)) = Qqg(1:nTq,1:nT);

    % Assign outputs
    varargout = {y1,y2,y3,y4};
  case 17
    % Derivatives of inequality constraints
    % [DQCGDF0,DQCGDF1      ] = BF(17,PAR, ~,F0,F1           )

    % Handle inputs
    [~,FA,FB] = deal(varargin{:});

    % number of constraints
    nc = min(nP,2);
    % Constraint for positive pressure
    FBA = FB - FA;
    dQcgdFB = [0,1,0;...
               0,1,2*FBA];
    dQcgdFA = -dQcgdFB;
    % Dispatch to 1...ng
    y1 = zeros(nc,ng); % dQcgdFA
    y2 = zeros(nc,ng); % dQcgdFB
    y1(1:nc,1:nP) = dQcgdFA(1:nc,1:nP);
    y2(1:nc,1:nP) = dQcgdFB(1:nc,1:nP);

    % Assign outputs
    varargout = {y1,y2};
  otherwise
    % Call generic bfct
    [varargout{1:nargout}] = bfct(mode,Pbfct,varargin{:});
end
end

function [gQ,IgQ] = bfab_fN(P, FQ)
% BFAB_FN Compute BFAB normalized basis functions

FQ = FQ(:);
% Compute normalized basis functions
 gQ = [(FQ-1),...
       (FQ-1).*FQ,...
       (FQ-1).*FQ.*(2*FQ - 1)];
% Compute normalized basis functions primitives
IgQ = [(FQ-1).^2/2,...
       (FQ-1).^2/6.*(FQ*2 + 1),...
       (FQ-1).^2/2.*(FQ).^2];
% Dispatch to 1...ng
 gQ =  gQ(:,[1:P(1),1:P(2)]);
IgQ = IgQ(:,[1:P(1),1:P(2)]);
end

function [alphapg,alphag] = bfab_fag(P, FBA)
% BFAB_FAG Conversion factors from normalized to physical basis functions

alphapg = FBA.^([1:P(1),1:P(2)].');
alphag  = FBA.*alphapg;
end

function [dgQ,dIgQ] = bfab_dfN(P, FQ)
% BFAB_DFN Compute derivatives of BFAB normalized basis functions

FQ = FQ(:);
% Compute normalized basis functions derivatives
 dgQ = [ones(size(FQ)),...
        2*FQ - 1,...
        (2*FQ - 1).^2 + 2*(FQ-1).*FQ];
% Compute derivatives of normalized basis functions primitives
dIgQ = [(FQ-1),...
        (FQ-1).*FQ,...
        (FQ-1).*FQ.*(2*FQ - 1)];
% Dispatch to 1...ng
 dgQ =  dgQ(:,[1:P(1),1:P(2)]);
dIgQ = dIgQ(:,[1:P(1),1:P(2)]);
end

function [dalphapg,dalphag] = bfab_dfag(P, FBA)
% BFAB_DFAG Derivatives of conversion factors from normalized to physical basis functions

u = [1:P(1),1:P(2)].';
dalphapg =  u   .*FBA.^(u-1);
dalphag  = (u+1).*FBA.^ u   ;
end
