function LY = meqsk(LY,k,LX)
%MEQSK substitute LY time slice
%
% LY = meqsk(LY,k,LX)
% Substitutes LY data at index k by LX slice(s)
% LX must have a number of times equal one, or to the number of indices k.
% k can be logical or numerical index
%
% See also: MEQIK, MEQXK
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if islogical(k); k=find(k); end

if numel(LX.t)==1 && numel(k)>1
  % copy single slice as often as needed
  LX = meqlpack(repmat(LX,1,numel(k)));
end
if numel(LX.t)~=numel(k)
  error('number of slices in LXt (%d) does not match indices in k (%d)',numel(LX.t),numel(k))
end

for ik=1:numel(k)
  
  for ifield = fieldnames(LX)'
    myfield = ifield{:};
    switch myfield
      case {'shot','tokamak'}
        % do nothing
      case {'Fx','Iy','Opy'}
        LY.(myfield)(:,:,k(ik)) = LX.(myfield)(:,:,ik);
        % handle these directly
      otherwise
        
        LXfield = LX.(myfield);
        
        if ~isfield(LY,myfield)
          if k~=1, error('can not assign non-existing field other than at first index'); end
          LY.(myfield) = LXfield;
        else
          LYfield = LY.(myfield); % base size decision on target field
          if isa(LYfield,'logical'), defaultvalue = false;
          else, defaultvalue  = nan;
          end
          
          if isempty(LYfield)
            LY.(myfield) = LXfield;
          elseif ismatrix(LYfield)
            n = numel(LXfield(:,ik)); m = size(LY.(myfield),1);
            if n>m, LY.(myfield) = [LY.(myfield);repmat(defaultvalue, n-m,size(LY.(myfield),2))]; end % extend original LY field with nans
            LY.(myfield)(:,k(ik)) = [LXfield(:,ik);repmat(defaultvalue, m-n,1)]; % pad with NaNs
          elseif ndims(LYfield) == 3
            LY.(myfield)(:,:,k(ik)) = LXfield(:,:,ik);
          elseif ndims(LYfield) == 4
            LY.(myfield)(:,:,:,k(ik)) = LXfield(:,:,:,ik);
          else            
            error('don''t know what to do');
          end
        end
    end
  end
end
