function [psstate,Vact] = meqps(G,psstate,dt,Vcmd)
% MEQPS - General power supply model with voltage saturations;
% alarms, and delays.
%
% [psstate] = meqps(G,[],dt,Vreq); % initialize power supply state
% [psstate,Vact] = meqps(G,psstate,dt,Vreq); % step state
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if nargin<4
  Vcmd = zeros(G.na,1); % default init
end

if ~isfield(G,'Vadelay')
  Vadelay = zeros(G.na,1); % default if none defined
else
  Vadelay = G.Vadelay;
end

if (dt~=0)
  idelays = round(Vadelay/dt);
else
  % case called with dt=0 (one time step only, delay is not relevant)
  idelays = zeros(size(Vadelay));
end

%% Saturate input voltage
Vsat = max(min(Vcmd,G.Vamax),G.Vamin);

if isempty(psstate)
    % initialize circular delay buffer as power supply state
    psstate = repmat(Vsat,1,max(idelays)); % initialize buffer with initial Vsat value
    Vact = Vsat; % init
else 
    %% Delays
    [Vact,psstate] = fifo_buffer(idelays,Vsat,psstate);
end
end


function [skd,buffer] = fifo_buffer(idelays,sk,buffer)

skd = zeros(size(sk));

for iu = 1:numel(sk)
  if idelays(iu)==0
    skd(iu) = sk(iu); % pass directly
  else
    skd(iu) = buffer(iu,idelays(iu));
  end
end

buffer = [sk,buffer(:,1:end-1)]; % drop last one, add new sk at front
end