function meqoptdebug(doplot,dotext,dohead)
% MEQOPTDEBUG Displays debug text and plots for MEQOPT
%
%  MEQOPTDEBUG(DOPLOT,DOTEXT[,DOHEADER])
%
% Inputs: doplot: do a plot
%         dotext: display text
%         dohead: display text header
%
% See also MEQOPT
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if ~(doplot || dotext), return; end
if nargin==2, dohead = false; end

% List of necessary variables from meqopt workspace, avoids shared variables and static workspaces
vars = evalin('caller','{L,LXs,LYs,Ra,IDs,shot,t,kit,tol,fgeF_failed}');
[L,LXs,LYs,Ra,IDs,shot,t,kit,tol,fgeF_failed] = deal(vars{:});

if kit>0 && ~fgeF_failed
  vars = evalin('caller','{resmaxF,resmaxRe,dxmax,dumax}');
  [resmaxF,resmaxRe,dxmax,dumax] = deal(vars{:});
else
  [resmaxF,resmaxRe,dxmax,dumax] = deal(NaN);
end

% Pack ouputs
LYs = meqlpack([LYs{:}]);
Ra = vertcat(Ra{:});
ID = vertcat(IDs{:});

% Extract from LY
rX  = LYs.rX;
zX  = LYs.zX;
rA  = LYs.rA;
zA  = LYs.zA;
nA  = LYs.nA;
rB  = LYs.rB;
zB  = LYs.zB;
Iy  = LYs.Iy;
Fx  = LYs.Fx;
Opy = LYs.Opy;
lB  = LYs.lB;
FB  = LYs.FB;
FA  = LYs.FA;
Ia  = LYs.Ia;
bp  = LYs.bp;
li  = LYs.li;
IpD = LYs.IpD;

% Misc.
nt = numel(t);
nA = min(nA,L.nD);
Ip = sum(reshape(Iy,[],nt),1);
sIp = sign(Ip);

% Plasma extent and elongation
ie0 = zeros(L.nD,nt);
for it=1:nt
  for iD=1:nA(it)
    b0 = bboxmex(Opy(:,:,it)==iD,L.zy,L.ry);
    ie0(iD,it) = (b0(4) - b0(2)) / (b0(3) - b0(1));
  end
end

% Get dz for LIU
if strcmp(L.code,'liu') && L.ndz
  dz = evalin('caller','dz');
  % Pad if needed
  dz(end+1:nA,:) = 0;
else
  dz = zeros(L.nD,nt);
end

% display information
if dohead
  meqoptthead();
end

if dotext
  meqopttdisp(L.P.tok,shot,t,rA,zA,dz,ie0,Ip,bp,li,kit,tol,resmaxF,resmaxRe,dxmax,dumax);
end

% optional plots
if doplot && nt>1
  % multi-slice MEQOPT debugger
  plotfun = @meqopt_debug_plot_multislice;
  clf; meqplotseq(L,LYs,'plotfun',plotfun,'LX',LXs);
  drawnow;
elseif doplot
  % single slice MEQOPT debugger

  % Fx contour levels
  if ~lB || ~sIp % if meqpdom could not find a valid LCFS point or Ip=0
    F = 21;
  else % plot specific contour lines depending on FB,FB-FA
    FA_ = sIp*max(sIp*FA);
    FB_ = sIp*min(sIp*FB);
    FBA = (FB_ - FA_);
    F = FB_-[linspace(-1,0,11), L.fq(end:-1:1),linspace(1,2.5,16) ]*FBA;
  end

  rztext = cell(1,nA);
  for iA = 1:nA
    rztext{iA} = sprintf('r,z=%5.3f,%+6.3fm%+4.1fmm:%d',rA(iA),zA(iA),dz(iA),sum(Opy(:) == iA));
  end

  % Symbols-lines to plot on the cross section
  %  1: Limiter contour
  %  2: Axes
  %  3: X-points
  %  4: Boundary points
  %  5: Iy contours/profiles
  %  6: meqpostq contours
  %  7: meqpostq gaps
  mvis = {'.c'              'om' 'xm' 'dm' '-w' '-m' '-oc'};
  rvis = {L.G.rl([1:end 1])  rA   rX   rB   []   []   []  };
  zvis = {L.G.zl([1:end 1])  zA   zX   zB   []   []   []  };

  if kit==1
    % on first iteration plot the contour lines of Iy
    [rvis{5},zvis{5}] = mycontourc(L.ry,L.zy,abs(Iy),linspace(.1,.95,5)*max(abs(Iy(:))));
  elseif nA
    % LIU used to show Tyg.*ag instead
    rvis{5} = L.ry;
    zvis{5} = sIp*Iy(iround(L.zy,zA),:)/max(abs(Iy(:)))*.2+zA;
  end

  % Flux contours, gaps
  if L.P.iterq && kit>1
    rq = LYs.rq;
    zq = LYs.zq;
    rvis{6} = rq([1:end 1],:);
    zvis{6} = zq([1:end 1],:);
    if L.nW
      aW = LYs.aW;
      rvis{7} = [L.G.rW L.G.rW-aW.*cos(L.G.oW)]';
      zvis{7} = [L.G.zW L.G.zW+aW.*sin(L.G.oW)]';
    end
  end

  % Bars
  bar_data = {};
  bar_col = {};
  bar_text = {};

  % Per code customization
  switch L.code
    case 'fbt'

      % Control points
      rH = LXs.gpr; zH  = LXs.gpz; bH  = logical(LXs.gpb);
      kH = ~isnan(rH) & ~isnan(zH) & ~isnan(bH);
      rH = rH(kH); zH  = zH(kH); bH = logical(bH(kH));
      bb = meqbbox(rH(bH),zH(bH),L.G.rl,L.G.zl,LXs.idoublet);

      % Symbols-lines to plot on the cross section
      %  8: Control points
      %  9: Bounding box
      mvis{8} = 'oy';
      rvis{8} =  rH;
      zvis{8} =  zH;
      mvis{9} = '-y';
      rvis{9} =  bb(:,[1,2,2,1,1]).';
      zvis{9} =  bb(:,[3,3,4,4,3]).';

      % Coil limits
      lim = L.P.limc*Ia;
      % Add margin factor to limits
      limu = L.P.limm.*L.P.limu;
      liml = L.P.limm.*L.P.liml;
      k = lim > 0;
      lim( k) =  lim( k)./limu( k);
      lim(~k) = -lim(~k)./liml(~k);
      % Current or Equation limit
      mask = sum(abs(L.P.limc)>0,2)<2;
      lima = lim(mask);
      limeq = lim(~mask);

      % Bars
      bar_data = {lima limeq Ra(ID==1) Ra(ID==2) Ra(ID==3) Ra(ID==4) Ra(ID==5) Ra(ID==6) Ra(ID==7) Ra(ID==8) Ra(ID==9)};
      bar_col  = {'b' 'g' 'c' 'c' 'y' 'r' 'k' 'b' 'g' 'y' 'm'};
      bar_text = {'Limits: Ia(b), Cia*Ia(g)' 'Cost fct: Ia(c) Coa*Ia(c) Da(y) Iu(r)' '\psi(k) B_{rzn}(b) C_{rzn}(g) V_{rr,rz,zz}(y) Va(m)'};
    case 'liu'

      % Symbols-lines to plot on the cross section
      %  8: FE locations
      %  9: Bounding box
      % 10: Optical measurements
      mvis{8} = 'oy';
      rvis{8} = L.rh;
      zvis{8} = L.zh;
      mvis{9} = '-y';
      rvis{9} = L.bh([1 3 3 1 1]);
      zvis{9} = L.bh([2 2 4 4 2]);
      if isfield(LXs,'ro') % Optical measurements
        mvis{10} = 'ok';
        rvis{10} = LXs.ro;
        zvis{10} = LXs.zo;
      end

      % Measurement residuals
      Ed = Ra(1:L.nd);
      chi = sqrt(sum(Ed.^2)/L.nd);
      Eq = Ra(L.nd+(1:L.nq));

      % Bars
      bar_data = {Ed(L.kdf) Ed(L.kdm) Ed(L.kda) Ed(L.kdu) Ed(L.kdt) Ed(L.kdp) Eq};
      bar_col  = {'r' 'b' 'c' 'k' 'y' 'm' [0 0.5 0]};
      bar_text = {'Residuals (fmastpq)' sprintf('X=%.1e',chi) sprintf('Convergence=%.1e',resmaxF)};
  end

  if nA > 1 && L.nD == 3 % display Ip per lobe
    tittext = sprintf('Ip=%.0f(%.0f/%.0f/%.0f)kA bp=%.2f li=%.2f',...
      Ip*1e-3,IpD(1)*1e-3,IpD(2)*1e-3,IpD(3)*1e-3,bp,li);
  else
    tittext = sprintf('Ip=%.0fkA bp=%.2f li=%.2f',Ip*1e-3,bp,li);
  end
    
  clf;
  meqvis([{sprintf('%s#%d %6.4fs/%d',L.P.tokamak,shot,t,kit)}...
    rztext...
    {tittext}],...
    L.G.rx,L.G.zx,Fx,F,FB,...
    rvis,zvis,mvis,...
    bar_data,bar_col,bar_text,...
    '')
  drawnow;
end

end

%% Text header display
function meqoptthead()
% display text header
nb=fprintf('%4s %6s %5s %8s %8s %5s %5s %3s %8s %8s %8s %5s %8s %8s %8s %8s %8s\n',...
  'tok','shot','kit','t[s]','Ip[kA]','bp','li','nA','rA[m]','zA[m]','dz[mm]','k','tol','resmaxF','resmaxRe','dxmax','dumax');
fprintf('%s\n',repmat('=',1,nb-1));
end

%% Text display
function meqopttdisp(tok,shot,t,rA,zA,dz,ie0,Ip,bp,li,kit,tol,resmaxF,resmaxRe,dxmax,dumax)
% debug display that works for multi-time slices and multi-domain

s=fprintf('%4s %6d %5d',tok,shot,kit);
for it = 1:numel(t)
  nA = numel(zA(:,it));
  % display information
  if it>1, fprintf('%*s',s,'');end % Add spaces for alignment
  nc = fprintf(' %8.4f %8.2f %5.2f %5.2f %3d ',t(it),Ip(:,it)*1e-3,bp(it),li(it),nA);
  for iA = 1:nA
    if iA > 1, fprintf('%*s',s+nc,'');end % Add spaces for alignment
    fprintf('%8.4f %+8.4f %+8.4f %5.2f ',rA(iA,it),zA(iA,it),dz(iA,it),1./ie0(iA,it));

    if iA==1 && it==1
      fprintf('%8.2e %8.2e %8.2e %8.2e %8.2e\n',tol,resmaxF,resmaxRe,dxmax,dumax)
    else
      fprintf('\n')
    end
  end
  if nA<1
    fprintf('%8s %8s %8s %5s ','','','','');
    fprintf('%8.2e %8.2e %8.2e %8.2e %8.2e\n',tol,resmaxF,resmaxRe,dxmax,dumax);
  end
end
end

%% Debug plot multislice
function hax = meqopt_debug_plot_multislice(hax,L,LX,LY)
  meqgplot(L.G,hax,'vl');
  if LY.nA == 0 % vacuum
    contour(hax,L.rx,L.zx,LY.Fx,11);
  else
    FFB = LY.F1(1:LY.nB);
    contour(hax,L.rx,L.zx,LY.Fx-FFB(1),11);
    contour(hax,L.rx,L.zx,LY.Fx,FFB,'k','linewidth',2);
  end

  plot(hax,LY.rX,LY.zX,'xk');
  plot(hax,LX.gpr,LX.gpz,...
    'LineStyle','none','Marker','o','MarkerEdgeColor','b','MarkerFaceColor','w');
  title(hax,meqtitle(L,LY,{'t','Ip'}))
end

