function [x,lambda,c_ok,s] = meqlsqlin(qpsolver,qpopts,A,b,Aiq,biq,Aeq,beq,lb,ub,xtyp,x0,ctol,no_ineq)
% MEQLSQLIN solve linear least squares problem in MEQ
%
% qpsolver: name of QP solver to use
% qpopts: options for QP solver
%
% min ||Ax - b||
% s.t. Aeq*x = beq
%      Aiq*x < ceq
%      lb < x < ub
%
% xtyp: typical x for scaling
% x0  : initial guess
% ctol: constraint tolerance
% no_ineq: disable inequality constraints
%
% x: solution
% lambda: lagrange multipliers for equality and inequality constraints
% c_ok: Flag whether constraints are satisfied (irrespective of no_ineq)
% s: status return (empty if ok)
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

s=[]; % status flag init
c_ok = false; % init

%% input sanity checks 
%  later make these optional?
if any(isnan(A)), error('nans in A'); end
if any(isnan(b)), error('nans in b'); end
if any(isnan(Aiq)), error('nans in Aiq'); end
if any(isnan(biq)), error('nans in biq'); end
if any(isnan(Aeq)), error('nans in Aeq'); end
if any(isnan(beq)), error('nans in beq'); end
if any(isnan(lb)), error('nans in lb'); end
if any(isnan(ub)), error('nans in ub'); end

%% Scalings
% Temporary rescaling aiming at x, cost function and constraints all being of order 1
xscale = 10.^round(log10( xtyp                         )); % state scaling (vector)
cscale = 10.^round(log10( max(sum(abs(  A.*xscale'),2)))); % residual scaling (scalar)
escale = 10.^round(log10(     sum(abs(Aeq.*xscale'),2) )); % exact constraints scale (vector)
iscale = 10.^round(log10(     sum(abs(Aiq.*xscale'),2) )); % inequality constraints scale (vector)

% catch case that all values are zero
xscale(xscale==0 | ~isfinite(xscale))=1;
cscale(cscale==0 | ~isfinite(cscale))=1;
escale(escale==0 | ~isfinite(escale))=1;
iscale(iscale==0 | ~isfinite(iscale))=1;

% cscale is empty if A is empty but should be a scalar
if isempty(cscale), cscale=1; end

% apply scalings
A_   =   A.*(xscale'./cscale);
b_   =   b./cscale;
Aiq_ = Aiq.*(xscale'./iscale);
biq_ = biq./iscale;
Aeq_ = Aeq.*(xscale'./escale);
beq_ = beq./escale;
lb_  = lb./xscale;
ub_  = ub./xscale;

%% Solve QP
H_ =  A_'*A_; 
f_ = -A_'*b_;

if no_ineq % disable inequality constraints
  [x_,~,stat,~,lambda] = qpsolver(H_,f_,  [],  [],Aeq_,beq_, [], [],x0,qpopts);
else
  [x_,~,stat,~,lambda] = qpsolver(H_,f_,Aiq_,biq_,Aeq_,beq_,lb_,ub_,x0,qpopts);
end

%% Solution

% If problem is infeasible, quadprog can return an empty solution
if stat <= 0 && isempty(x_), x_ = NaN(size(xtyp)); end

% Add lagrange coefficients for inequalities if needed
if no_ineq
  lambda.ineqlin = zeros(size(Aiq_,1),1);
  lambda.upper   = zeros(numel(x_),1);
  lambda.lower   = zeros(numel(x_),1);
end

% Rescale answer
x = x_.*xscale;
lambda.eqlin   = lambda.eqlin  .*cscale.^2./escale;
lambda.ineqlin = lambda.ineqlin.*cscale.^2./iscale;
lambda.upper   = lambda.upper  .*cscale.^2./xscale;
lambda.lower   = lambda.lower  .*cscale.^2./xscale;

%% QP checks - check on scaled quantities
eqok = (norm(Aeq_*x_-beq_,Inf) < ctol); % residual check
lbok = isempty(lb_) || ~any( (lb_ -ctol) > x_ ); % lower bound check
ubok = isempty(ub_) || ~any( x_ > (ub_ + ctol) ); % upper bound check
iqok = ~any( Aiq_*x_ > (biq_ + ctol)  ); % inequality constriant check

if (stat == 0)
  s.msgid = 'QPFail'; s.msg = 'QP solving failed: exceeded maximum iterations'; return;
elseif (stat == -2)
  s.msgid = 'QPFail'; s.msg = 'QP solving failed: infeasible'; return;
elseif (stat < 0)
  s.msgid = 'QPFail'; s.msg = 'QP solving failed'; return;
end

if ~eqok
  s.msgid = 'QPEqFail'; s.msg = 'QP result does not satisfy exact constraints'; return;
end

if ~no_ineq && ~lbok
  s.msgid = 'QPLBFail'; s.msg = 'QP result violates lower bound'; return
end

if ~no_ineq && ~ubok
  s.msgid = 'QPUBFail'; s.msg = 'QP result violates upper bound'; return
end

if ~no_ineq && ~iqok
  s.msgid = 'QPIneqFail'; s.msg = 'QP result violates inequality constraint'; return
end

% check for constraint satisfaction (irrespective of no_ineq)
c_ok = ( lbok && ubok && iqok );

end
