function [Ra,Re,Ri] = meqlsqeval(L,...
  Raxc,Rauc,Razc,Ra0c,Rexc,Reuc,Rezc,Re0c,Rixc,Riuc,Rizc,Ri0c,...
  d1dt,d2dt,xs,us,zs)
% MEQLSQEVAL evaluates the optimization residuals for MEQOPT
%
% A detailed description of the inputs and outputs is available in MEQOPT.
%
% This function computes the current residuals Ra/Re/Ri
%
% Time derivatives in the optimization problem:
%   For any function F(t), we use the approximations:
%     dFdt  (t(i)) = sum_j d1dt(i,j)*F(t(j))
%     d2Fdt2(t(i)) = sum_j d2dt(i,j)*F(t(j))
%   For cost function terms or constraints of the form
%     R(t(i)) = R0 + Rx*dxdt  (t(i)) + Ru*dudt  (t(i))
%   we substitute dxdt, dudt with the approximations above to get the
%   optimization problem formulation
%
% SEE ALSO: FBTHELP, FBTLSQINIT, MEQLSQPREP, MEQLSQSOLVE, MEQOPT, FBTT
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% sizes
[nd,nt] = size(Raxc);
nx = L.nN;

if nd>3
  error('Only up to 2nd order derivative constraints implemented')
end

% Init
Ra = cell(nd,nt);
Re = Ra;
Ri = Ra;

%% Compute right-hand sides and response matrices
% - Static case
if ~L.isEvolutive
  % Residual
  Ra{1} = Ra0c{1} - Raxc{1}*xs{1} - Rauc{1}*us{1} - Razc{1}*zs{1};
  Re{1} = Re0c{1} - Rexc{1}*xs{1} - Reuc{1}*us{1} - Rezc{1}*zs{1};
  Ri{1} = Ri0c{1} - Rixc{1}*xs{1} - Riuc{1}*us{1} - Rizc{1}*zs{1};
  return
end

% - Evolutive case
for it = 1:nt
  nu = numel(us{it});
  nz = numel(zs{it});
  for id = 1:nd
    if id == 1
      %   Value constraints
      x_ = xs {it};
      u_ = us {it};
      z_ = zs {it};
    else
      %   1st/2nd order time derivative constraints
      x_ = zeros(nx,1);
      u_ = zeros(nu,1);
      z_ = zeros(nz,1);
      % Loop over all slices that influence the time derivative at slice it
      for itt = 1:nt
        if id == 2
          fac = d1dt(it,itt);
        else
          fac = d2dt(it,itt);
        end
        if fac == 0, continue; end % Skip points not in the stencil
        x_ = x_ + fac*xs {itt};
        u_ = u_ + fac*us {itt};
        z_ = z_ + fac*zs {itt};
      end
    end
    
    % Residual
    Ra{id,it} = Ra0c{id,it} - Raxc{id,it}*x_ - Rauc{id,it}*u_ - Razc{id,it}*z_;
    Re{id,it} = Re0c{id,it} - Rexc{id,it}*x_ - Reuc{id,it}*u_ - Rezc{id,it}*z_;
    Ri{id,it} = Ri0c{id,it} - Rixc{id,it}*x_ - Riuc{id,it}*u_ - Rizc{id,it}*z_;
  end
end

end
