%MEQLPACK  Packs an array of structures as a structure of arrays
% Y = MEQLPACK(X,NOPACK) returns a structure with the fields of the array X
% concatenated along the 3rd dimension and squeezed, except those in NOPACK.
% Default for NOPACK: {'aq','rq','zq','ppQg','TTpQg','FW'};
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
function Y = meqlpack(X,nopack)

if nargin<2
  % these variables should not be packed.
  nopack={'aq','rq','zq','PpQg','TTpQg','FW'};
end

if numel(X) <= 1
  Y = X;
else
  % Discard slices with empty time field
  mask = ~cellfun(@isempty,{X.t});
  X = X(mask);
  % Loop through fields
  for k = fieldnames(X)'
    V = {X.(k{1})};
    catval = safecat4(V);
    Y.(k{1}) = cast(catval,class(X(1).(k{1})));
    if ~ismember(k,nopack)
      Y.(k{1}) = squeeze(Y.(k{1}));
      if isvector(Y.(k{1}))
        Y.(k{1}) = reshape(Y.(k{1}),1,[]);
      end
    end
  end
end
end

function catval = safecat4(V)
% Uses cat(4,...) for most cases except if the different time slices
% are not the same size (e.g. limited/diverted transitions)

[m,n,o] = getfieldsizes(V);
samesize = all(diff(m)==0) && all(diff(n)==0) && all(diff(o)==0);

catdim = 4;

if samesize
  catval = cat(catdim,V{:});
else
  nt = numel(V);
  if islogical(V{1})
    catval = false(max(m),max(n),max(o),nt); % default false for logicals
  else
    catval =   NaN(max(m),max(n),max(o),nt); % default NaNs otherwise
  end
  % fill each time index with field value
  for it=1:nt
    catval(1:m(it),1:n(it),1:o(it),it) = V{it};
  end
end
end

function [m,n,o] = getfieldsizes(V)
nt = numel(V);
m = zeros(nt,1);
n = m;
o = m;
for ii=1:nt
  % Use 4 output arguments to get actual size of first 3 dimensions
  [m(ii),n(ii),o(ii),~] = size(V{ii});
end
end
