function [Tstate,LY,alarm] = meqlim(L,LY,dt,Tstate)
% function [Tstate,LY,alarm] = meqlim(L,LY,dt,it,Tstate)
% Checks coil system operational limits
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if isempty(Tstate)
  Tstate = zeros(L.G.na,1); % init
else
  Tstate = meqtherm(LY.Ia,dt,Tstate);
end

%% Check coil limits

% Coil current limit
[Iarel,Ialarm] = meqIalim(L.G,LY.Ia);

% Thermal limit
Tarel = Tstate./L.G.Talim;
Talarm = (Tarel > 1);

% Coil combination equation limit
if isfield(L.P,'eqfct')
  % optional equation or coil current limits
  [Parel,Palarm] = L.P.eqfct(L.G,LY.Ia);
else
  Parel = zeros(0,1); Palarm = false;
end

%% Alarms
if any(Ialarm) && (L.P.debug || L.P.stoponalarm)
  fprintf('\n*** Current limit exceeded on coil %s',L.G.dima{Ialarm});
  fprintf('\n');
end
if any(Palarm) && (L.P.debug || L.P.stoponalarm)
  fprintf('\n*** Protection limit equation %d violated',find(Palarm));
  fprintf('\n');
end
if any(Talarm) && (L.P.debug || L.P.stoponalarm)
  fprintf('\n*** Thermal limit on coil %d violated',L.G.dima{Talarm});
  fprintf('\n');
end

alarm = any(Talarm) || any(Ialarm) || any(Palarm);

%% Outputs
LY.Tarel = Tarel;
LY.Iarel = Iarel;
LY.Parel = Parel;
end

function [Iarel,Ialarm] = meqIalim(G,Ia)
% [Iarel,Ialarm] = meqIalim(G,Ia)
% Iarel: Relative violation of coil current limits
% Ialarm: Current limits violation

% Check current limits
Ialarm = Ia>G.Iamax | Ia<G.Iamin;
Iarel = max(Ia./G.Iamax,Ia./G.Iamin);

end

function [Tstate] = meqtherm(Ia,dt,Tstate)
% Thermal protection implementation for coil currents
% Assume dissipated power: R*I^2.
% Thermal limit implemented as time integral of I.^2.

Tstate = Tstate + dt*Ia.^2; % time integral of current squared

end