function LX = meqinterp(LY,t,method,allowInf)
%MEQINTERP Interpolate LY fields on to time basis t.
% LX = meqinterp(LY,t,method[,allowInf]);
% method can be 'linear' (default),'previous','next', 'nearest', or
%    anything accepted by INTERP1
% allowInf, if set to true, allows interpolation from Infinite to finite
%   quantities using the reciprocal 1/interp(x,1./y,xx).
%   If y contains both 0 and Inf in some interval, then the mid-point is set to 1
%   allowInf can not be used in combination with other interpolation
%   methods than those listed above
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if nargin<3
  method = 'linear';
end

if nargin<4
  allowInf = false;
else
  assert(~allowInf || any(strcmp(method,{'linear','previous','next', 'nearest'})),...
    'Can''t use allowInf=true in combination with interpolation method %s',method)
end

switch method
  case 'nearest'
    LX = meqxk(LY,iround(LY.t,t));
  case 'previous'
    ii=ifloor(LY.t,t); 
    assert(~any(ii==0),'meqinterp:OutOfTimeRange',...
      'Requested time %f before first time %f with method=''previous''',t(find(ii==0,1,'first')),min(LY.t))
    LX = meqxk(LY,ii);
  case 'next'
    ii=iceil(LY.t,t); 
    assert(~any(ii==0),'meqinterp:OutOfTimeRange',...
      'Requested time %f before last time %f with method=''next''',max(LY.t),t(find(ii==0,1,'first')))
    LX = meqxk(LY,ii);
  otherwise
    LX = meqinterp1(LY,t,method,allowInf);
end

end

function LX = meqinterp1(LY,t,method,allowInf)
assert( max(t)<=max(LY.t) && min(t)>=min(LY.t),'meqinterp:OutOfTimeRange',...
  'Range t [%2.2f,%2.2f] exceeds range of LY.t [%2.2f,%2.2f]',min(t),max(t),min(LY.t),max(LY.t));

nt_ = numel(LY.t);
% Shortcut when LY has a single time slice
if nt_==1
  LX = meqlpack(repmat(LY,1,numel(t)));
end

% Fields without interpolation
nointerp = {'shot'};
for field=intersect(fieldnames(LY)',nointerp)
  LX.(field{:}) = LY.(field{:})(:,1); % just take first
end

% Complete shortcut
if nt_==1
  return
end

for field=setdiff(fieldnames(LY)',nointerp)
  val = LY.(field{:});
  siz = size(val);
  timedim = find(siz==nt_,1,'last');
  if isempty(timedim) % Does not depend on time
    LX.(field{:}) = val; % pass direct
  elseif isempty(val) && (isnumeric(val) || islogical(val))
    % Keep the original shape up to timedim
    LX.(field{:}) = zeros([siz(1:timedim-1),numel(t)],class(val));
    % (timedim has to be >1 as timedim should be the last dimension and be a non-empty dimension)
  elseif isempty(val)
    error('don''t know how to treat empty field with class %s and timedim=%d',class(val),timedim);
  elseif ~isnumeric(val)
    LX.(field{:}) = meqinterp_nonnumeric(LY.t,val,t,timedim,field{:});
  else % numeric and ~empty: interpolate
    % permute since time dimension must be first in interp1 call
    p=[timedim,1:timedim-1];
    LX.(field{:}) = ipermute(interp1meq(LY.t',permute(val,p),t',method,allowInf),p);
  end
end
end

function yy = interp1meq(x,y,xx,method,allowInf)
% generalized interp1 so that it can interpolate also between Inf values
% and finite values, it does this by interpolating the inverse and then
% inverting again.
if ~any(isinf(y)) 
  % default interp1
  yy = interp1(x,y,xx,method);
else
  assert(allowInf,'meqinterp:NotAllowInf','set allowInf=true to allow interpolations involving Inf values');
  % treat inf cases
  sy = size(y); nx = sy(1);
  y = reshape(y,nx,[]); % Transform to a 2D array
  yy = zeros(size(xx,1),size(y,2));
  for ii=1:numel(x)-1 % treat each x interval between two x points separately
    j1 = ii; j2 = ii+1; jj = [j1,j2];
    kk = (xx>=x(ii) & xx<=x(ii+1)); % interested indices in xx
    noinf = ~any(isinf(y(jj,:)));
    iinf0 = any(isinf(y(j1,:)) & (y(j2,:)==0),1); % [Inf,0] value pattern
    i0inf = any(isinf(y(j2,:)) & (y(j1,:)==0),1); % [0,Inf] value pattern 
    iinfN = ~noinf & ~(iinf0|i0inf);
    
    if any(noinf) % non-inf parts, as usual
      yy(kk,noinf) = interp1(x(jj),y(jj,noinf),xx(kk),method);
    end
    if any(iinfN) % Interpolate between inf and finite value, use reciprocal
      yy(kk,iinfN) = 1./interp1(x(jj),1./y(jj,iinfN),xx(kk),method);
    end
    if any(iinf0 | i0inf) % Interpolate between inf and zero value, use reciprocal and insert value 1 at mid-point
      % part before midpoint
      x1 = [x(j1);mean(x(jj))]; % insert mid point
      y1 = [y(j1,:);ones(size(y(j1,:)))];
      k1 = xx>=x(ii) & xx<x1(2); % interested xx indices
      
      % use reciprocal here for iinf0, normal for i0inf
      if any(iinf0), yy(k1, iinf0) = 1./interp1(x1,1./y1(:,iinf0),xx(k1),method); end
      if any(i0inf), yy(k1, i0inf) =    interp1(x1,   y1(:,i0inf),xx(k1),method); end      
      % part after midpoint
      x2 = [x1(2);x(j2)];
      y2 = [ones(size(y(j2,:)));y(j2,:)];
      k2 = xx>=x1(2) & xx<=x(ii+1);

      % use reciprocal here for i0inf, normal for iinf0
      if any(iinf0), yy(k2, iinf0) =    interp1(x2,   y2(:,iinf0),xx(k2),method); end
      if any(i0inf), yy(k2, i0inf) = 1./interp1(x2,1./y2(:,i0inf),xx(k2),method); end
    end
  end
  yy = reshape(yy,[numel(xx),sy(2:end)]); % Restore y original dimensions
end
  
end

function val2 = meqinterp_nonnumeric(t1,val1,t2,timedim,fldname)
% Interpolate non-numeric values
if ischar(val1), val2=val1; return; end % do not interpolate
if isa(val1,'logical')
  nt1 = numel(t1); nt2 = numel(t2);
  sz1 = size(val1);
  val1 = reshape(val1,[],nt1); % Reshape to 2D array
  val2 = false(size(val1,1),nt2); % init
  for it2=1:nt2
    % find interpolation intervals -interpolating between slices i1 and i2
    i1 = ifloor(t1,t2(it2));
    i2 = iceil(t1,t2(it2));
    % check that both sides are equal, otherwise throw error
    %   this covers the case where t2(it2) is a member of t1 and i1==i2
    if ~isequal(val1(:,i1),val1(:,i2))
      msgid = 'meqinterp:interpNotAllowed';
      error(msgid,'%s %d and %d (t=%4.3f,%4.3f) have logical but different values, can not interpolate to t=%4.3f',...
        fldname,i1,i2,t1(i1),t1(i2),t2(it2))
    else
      val2(:,it2) = val1(:,i1);
    end
  end
  val2 = reshape(val2,[sz1(1:timedim-1),nt2]); % Restore dimensions
else
  error('don''t know how ot treat class %s',class(val1));
end
end