function [IpQ,WkQ,WpQ,VpQ,FtQ,Ft0Q,OpQ] = meqintQ(L,F0,F1,rBt,ag,Fx,Opy)
%MEQINTQ Computes integral quantities on pQ grid.
% [IpQ,WkQ,WpQ,VpQ,FtQ,Ft0Q,OpQ] = meqintQ(L,FA,FB,rBt,ag,Fx,Opy)
% see MEQINT
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if all(ag==0)
  IpQ  = zeros(L.nQ,L.nD);
  WkQ  = IpQ;
  WpQ  = IpQ;
  VpQ  = IpQ;
  FtQ  = IpQ;
  Ft0Q = IpQ;
  OpQ  = IpQ;
  return;
end

% Check Opy
Opy(Opy>L.nD) = 0;

% Normalized flux grid
FQ = L.pQ.^2;

% r and 1/r on y grid
rry = L.rry(:);
irry = 1./rry;
masky = (Opy(:)>0);
rry =   rry.*masky;
irry = irry.*masky;

% For Ip/Wk/Ft integrands
[G,IG] = L.bfct(91,L.bfp,Fx(2:end-1,2:end-1),F0,F1);
% Combine with r or 1/r
fac = (rry.*L.fPg.' + irry.*L.fTg.');
fac = fac.*L.TDg(max(1,Opy(:)),:); % Deselect basis functions that are inactive in certain domains
Tyg  = fac.* G;
ITyg = fac.*IG;

% For Wp integrand
RBp2y = ((Fx(2:end-1,3:end  ) - Fx(2:end-1,1:end-2))./(4*pi*L.drx)).^2 +  ...
        ((Fx(3:end  ,2:end-1) - Fx(1:end-2,2:end-1))./(4*pi*L.dzx)).^2;

% CIZRMEX performs integration over the plasma cross-section and
% automatically bins result according to radial grid and domain

% Here we define each integrand
Vy = zeros(L.ny,7);
Vy(:,1) = Tyg*(L.idsx*ag);                  % IpQ
Vy(:,2) = ITyg*(L.idsx*ag.*L.fPg)*1.5;      % WkQ
Vy(:,3) = RBp2y(:).*irry/4e-7;              % WpQ
Vy(:,4) = 2*pi*rry;                         % VpQ
Vy(:,5) = ITyg*(ag.*L.fTg*2e-7/rBt*L.idsx); % FtQ
Vy(:,6) = rBt.*irry;                        % Ft0Q
Vy(:,7) = repmat(L.idsx,L.ny,1);            % OpQ

% Perform integration
IVQD = cizrmex(Fx,Opy,Vy,L.dsx,FQ,F0,F1);

% Dispatch
IpQ  = IVQD(:,:,1);
WkQ  = IVQD(:,:,2);
WpQ  = IVQD(:,:,3);
VpQ  = IVQD(:,:,4);
FtQ  = IVQD(:,:,5);
Ft0Q = IVQD(:,:,6);
OpQ  = IVQD(:,:,7);

end
