function meqgplot(G,ax,opts)
%MEQGPLOT: Generic geometry structure plotting
% meqplot(G) % plot G structure
% meqplot(G,ax); % pass axis handle
% meqplot(G,ax,options); % pass options to plot
%   options: v: Vessel
%            l: limiter contour
%            m: B probles
%            f: flux loops
%            W: wall gaps
%            w: coil windings
%            x: grid box
%
% typing 'legend show' after the plotting will show a legend
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if nargin<3
  opts = 'tvlmfWwx';
end

if nargin<2 || isempty(ax)
  ax = gca;
end
 
assert(ishandle(ax),'invalid handle supplied');
 
if contains(opts,'x')
  % grid box
  [rrx,zzx] = meshgrid(G.rx,G.zx);
  mesh(rrx,zzx,zeros(size(rrx)),'edgecolor',0.8*[1 1 1],'displayname','grid')
end
hold(ax,'on');

% vessel
if contains(opts,'v')
  % plot per component
  for in = 1:size(G.Tnv,1)
    seln = G.Tnv(in,:)~=0;
    plot(ax,G.rv(seln),G.zv(seln),'.b','displayname',G.dimn{in});
  end
end
 
% limiter
if contains(opts,'l')
  rl = [G.rl;G.rl(1)]; zl = [G.zl;G.zl(1)];
  plot(ax,rl,zl,'color',0.3*[1 1 1],'linewidth',1.5,'displayname','limiter');
end
 
if contains(opts,'w')
  % windings and coils
  plot(ax,G.rw,G.zw,'rs','displayname','coil winding');
end
 
if contains(opts,'f')
  plot(ax,G.rf,G.zf,'xm','displayname','flux loops');
end

if contains(opts,'m')
  plot(ax,G.rm,G.zm,'om','displayname','B probes');
end
 
% gaps
if contains(opts,'W') && G.nW
  plot(ax,[G.rW G.rW-G.aW.*cos(G.oW)]',...
          [G.zW G.zW+G.aW.*sin(G.oW)]','-o','displayname','wall gap');
end
 
view(2);
axis(ax,'equal','tight');
 
end
