function [dIydx, dFxdx, dFAdx, dFBdx, drAdx, dzAdx, dFndx, dBrndx, dBzndx, dFtdx, dIpdx, dbpdx, dbpDdx, dlidx, dqAdx, dWkdx, dSdx] = meqdFdI(L,LY)
% Evaluate derivatives of post-processing outputs of F operator
% w.r.t. changes in the non-linear state
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% Enable computation of signeo derivative. 
% Since it involves expensive meqintQ, done only when necessary
dsig = nargout > 16; 

assert(numel(LY.t)==1,'dFdI must be called for 1 timeslice only, use mexqk() to slice')

% Use analytical jacobian if possible
if L.P.anajac && ~dsig
  [dIydx, dFxdx, dFAdx, dFBdx, drAdx, dzAdx, dFndx, dBrndx, dBzndx, dFtdx, dIpdx, dbpdx, dbpDdx, dlidx, dqAdx, dWkdx] = dFdI_anajac(L,LY);
  return
end

% Function arguments
x = L.LX2x(LY);
rBt = LY.rBt;
signeo = LY.signeo;

% Identify algoNL variant
%   xHasIy: state has Iy
xHasIy = strcmpi(L.P.algoNL,'all-nl');

fun = @evalFxy;
x0 = {L,x,rBt,signeo,xHasIy};
sizes = {L.ny,L.nx,LY.nA,LY.nB,LY.nA,LY.nA,L.nn,L.nn,L.nn,1,1,1,L.nD,1,LY.nA,1,LY.nB};

% Derivative w.r.t. changes in x
epsval = sqrt(eps);
if dsig
  [dIydx, dFxdx, dFAdx, dFBdx, drAdx, dzAdx, dFndx, dBrndx, dBzndx, dFtdx, dIpdx, dbpdx, dbpDdx, dlidx, dqAdx, dWkdx, dSdx] = ...
    jacfd(fun,x0,'iargin',2,'epsval',epsval,'szF0',sizes);
else
  [dIydx, dFxdx, dFAdx, dFBdx, drAdx, dzAdx, dFndx, dBrndx, dBzndx, dFtdx, dIpdx, dbpdx, dbpDdx, dlidx, dqAdx, dWkdx] = ...
    jacfd(fun,x0,'iargin',2,'epsval',epsval,'szF0',sizes(1:end-1));
end

% Set derivatives of bpD to 0 for domains where IpD=0 (singular point)
dbpDdx(LY.IpD==0,:) = 0;
    
end

function [Iy,Fx,FA,FB,rA,zA,Fn,Brn,Bzn,Ft,Ip,bp,bpD,li,qA,Wk,S] = evalFxy(L,x,rBt,signeo,xHasIy)
sig = nargout > 16;
if sig && ~isequal(size(signeo),[L.nQ,L.nD])
  error('meqdFdI:signeoSize','Conductivity integral derivative required but size(signeo)=[%d,%d] instead of [L.nQ,L.nD]',size(signeo,1),size(signeo,2));
end

% Extract NL state
ixe = [L.ind.ixa,L.ind.ixu];
xSI = x.*L.xscal;
if xHasIy
  Iy = reshape(xSI(L.ind.ixGS),L.nzy,L.nry);
else
  Fx = reshape(xSI(L.ind.ixGS),L.nzx,L.nrx);
end
ag = xSI(L.ind.ixg );
Ie = xSI(      ixe );

% Compute missing Fx or Iy
if xHasIy
  Fx = meqFx(L,Iy,Ie);
else
  Iyie = L.Tye*Ie;
  Iy = -(L.dlst*Fx(:))./L.rhsf - Iyie;
end

% Identify plasma domain
sIp = sign(sum(Iy(:)));
[rA,zA,FA,dr2FA,dz2FA,drzFA,~,~,~,~,~,~,...
  ~,~,FB,~,~,Opy,F0,F1,pdomstat,msg,id] = meqpdom(Fx,sIp,L.P.isaddl,L);
if ~pdomstat, meqmsge('e',mfilename,L.P.tok,NaN,NaN,L.P.shot,msg,id); end

% Compute interpolated fluxes and fields
if L.nn
  [Fn,Brn,Bzn,~,~,~,~] = L.P.infct(L.rx,L.zx,Fx,L.P.rn,L.P.zn,L.inM);
else
  Fn = [];  Brn = []; Bzn = [];
end
  
nB = numel(FB);

% Total integrals
[~,TpDg,ITpDg] = L.bfct(1,L.bfp,Fx,F0,F1,Opy,L.ry,L.iry);
[Ip,Wk,~,~,~,~,~,Ft,~,bp,~,~,li,~] = ...
  meqint(L.fPg,L.fTg,TpDg,ITpDg,ag,Fx,Opy,L,rBt);
bpD = zeros(L.nD,1);
for iD=1:L.nD
  [~,~,~,~,~,~,~,...
    ~,~,bpD(iD),~,~,~,~] = ...
    meqint(L.fPg,L.fTg,TpDg(iD,:),ITpDg(iD,:),ag,Fx,int8(Opy==iD),L,rBt);
end

% q axis using small diamagnetic approximation (LIUQE 2015 paper eq. 88)
nA = int32(numel(FA));
qA = zeros(nA,1);
[gAg,IgAg] = L.bfct(5,L.bfp,[],F0,F1); IgAg = 2e-7*L.idsx*IgAg;
for iA=1:nA
  TyAg = gAg(iA,:).*(rA(iA)*L.fPg+L.fTg/rA(iA)).'; % TyAg: Axis current (jphi.dR.dZ) contribution from each basis function
  cqA = 4e-7*pi*rA(iA)^2*rBt*sqrt(dr2FA(iA)*dz2FA(iA)-drzFA(iA)^2)/(abs(dr2FA(iA)+dz2FA(iA))*L.dsx);
  qA(iA) = (rBt.^2+(IgAg(iA,:).*L.fTg.')*ag)/(cqA*TyAg*ag);
end

% Neoclassical conductivity integral
if sig
  [~,~,~,~,FtQ,Ft0Q] = meqintQ(L,F0,F1,rBt,ag,Fx,Opy);
  FtPQ  = FtQ+Ft0Q; % total Plasma toroidal flux on pQ grid
  
  % CDE conductivity integral
  S = zeros(1,nB);
  for iD=1:nB
    [~,~,~,TQ] = ...
      meqprof(L.fPg,L.fTg,ag,L.pQ(:).^2,F0,F1,rBt,L.bfct,L.bfp,L.idsx);
    iTsQ = signeo(:,iD)./TQ(:,iD).^2; % sigma/F^2 profile appearing in various places
    S(:,iD) =  rBt/2/pi * trapz(FtPQ(:,iD), iTsQ);
  end
end

end

function [dIydx, dFxdx, dFAdx, dFBdx, drAdx, dzAdx, dFndx, dBrndx, dBzndx, dFtdx, dIpdx, dbpdx, dbpDdx, dlidx, dqAdx, dWkdx] = dFdI_anajac(L,LY)

IpD   = LY.IpD;
Ip    = LY.Ip;
rBt   = LY.rBt;
ag    = LY.ag;
nA    = LY.nA;
rA    = LY.rA;
zA    = LY.zA;
dr2FA = LY.dr2FA;
dz2FA = LY.dz2FA;
drzFA = LY.drzFA;
nB    = LY.nB;
Fx    = LY.Fx;
Opy   = LY.Opy;
bpD   = LY.bpD;
bp    = LY.bp;
li    = LY.li;

sIp = sign(Ip);

% Identify algoNL variant
%   xHasIy: state has Iy
xHasIy = strcmpi(L.P.algoNL,'all-nl');

% Call meqpdom to get derivatives of FA/FB
[~,~,~,~,~,~,~,~,~,~,~,~,...
  ~,~,~,~,~,~,F0,F1,pdomstat,msg,id,dF0dFx,dF1dFx,ixI] = meqpdom(Fx,sIp,L.P.isaddl,L);
if ~pdomstat, meqmsge('e',mfilename,L.P.tok,NaN,NaN,L.P.shot,msg,id); end

% Additional derivatives
if L.P.icsint
  [dFAdFx, dzAdFx, drAdFx, ddz2FAdFx, ddr2FAdFx, ddrzFAdFx] = ...
    asxycsJac(Fx,zA,rA,dz2FA,dr2FA,drzFA,L);
else
  [dFAdFx, dzAdFx, drAdFx, ddz2FAdFx, ddr2FAdFx, ddrzFAdFx] = ...
    asxyJac(Fx, L.dzx, L.drx, L.idzx, L.idrx, ixI(1:nA));
  % This avoids problems in Octave with singleton expansion operations for sparse matrices
  drAdFx    = full(   drAdFx);
  ddz2FAdFx = full(ddz2FAdFx);
  ddr2FAdFx = full(ddr2FAdFx);
  ddrzFAdFx = full(ddrzFAdFx);
end
dFBdFx = dF1dFx(1:nB,:);

% Prerequisites
[Tyg,TpDg,ITpDg] = L.bfct(1,L.bfp,Fx,F0,F1,Opy,L.ry,L.iry);
[g0g,Ig0g]     = L.bfct(5,L.bfp,Fx,F0,F1);
gAg  =  g0g(1:nA,:);
IgAg = Ig0g(1:nA,:);

%   Derivatives of TpDg, ITpDg
[dTygdFy,dTygdF0,dTygdF1,dITygdF0,dITygdF1] = L.bfct(11,L.bfp,Fx,F0,F1,Opy,L.ry,L.iry);

[~,~,~,dTpDgdFx ] = meqbfct1Jac(L,nB,Opy,dTygdFy, dTygdF0, dTygdF1,dF0dFx,dF1dFx);
[~,~,~,dITpDgdFx] = meqbfct1Jac(L,nB,Opy, Tyg   ,dITygdF0,dITygdF1,dF0dFx,dF1dFx);

%   Derivatives of ITpg (sum of ITpDg over domains)
dITpgdFx = reshape(sum(reshape(dITpDgdFx,L.nD,L.ng,L.nx),1),L.ng,L.nx);
ITpg     =         sum(         ITpDg                   ,1);

%   Derivatives of Wp
dWpdFx = vizrJac(Fx,Opy,L.iry,L.drx,L.dzx);
%   Derivatives of gAg/IgAg
[dg0gdF0,dg0gdF1,dIg0gdF0,dIg0gdF1] = L.bfct(15,L.bfp,[],F0,F1);
dgAgdFx  = reshape(reshape( dg0gdF0(1:nA,:,:),nA*L.ng,L.nD)*dF0dFx + reshape( dg0gdF1(1:nA,:,:),nA*L.ng,L.nD)*dF1dFx,nA,L.ng,L.nx);
dIgAgdFx = reshape(reshape(dIg0gdF0(1:nA,:,:),nA*L.ng,L.nD)*dF0dFx + reshape(dIg0gdF1(1:nA,:,:),nA*L.ng,L.nD)*dF1dFx,nA,L.ng,L.nx);
%   Derivatives of trH/detH
dtrHdFx  = ddr2FAdFx + ddz2FAdFx;
ddetHdFx = (dz2FA.*ddr2FAdFx + dr2FA.*ddz2FAdFx - 2*drzFA.*ddrzFAdFx);

% Derivatives of Ft
Ftg = 2e-7/rBt*(     L.fTg .'.* ITpg   );
Ftx = 2e-7/rBt*((ag.*L.fTg).' *dITpgdFx);

% Derivatives of IpD
IpDg =                      TpDg;
IpDx = reshape(sum(reshape(dTpDgdFx,L.nD,L.ng,L.nx).*ag.',2),L.nD,L.nx);

% Derivatives of Ip (total)
Ipx = sum(IpDx,1);
Ipg = sum(IpDg,1);

% Derivatives of pbarD = 2/3*WkD 
pbarDx = reshape(sum(reshape(dITpDgdFx,L.nD,L.ng,L.nx).*(L.fPg.*ag).',2),L.nD,L.nx);
pbarDg =                      ITpDg                   .*(L.fPg    ).';

% Derivatives of bpD
WND  = 1e-7*pi*L.P.r0*IpD.^2;
iWND = 1./WND; iWND(WND==0) = 0;
iIpD = 1./IpD; iIpD(IpD==0) = 0;
bpDx = pbarDx.*iWND - 2*bpD.*iIpD.*IpDx;
bpDg = pbarDg.*iWND - 2*bpD.*iIpD.*IpDg;

% Derivatives of bp (total)
WN  = 1e-7*pi*L.P.r0*Ip^2;
bpx = ((ag.*L.fPg).' *dITpgdFx)./WN - 2*bp/Ip*Ipx;
bpg = (     L.fPg .'.* ITpg   )./WN - 2*bp/Ip*Ipg;

% Derivatives of li (total)
lix = dWpdFx./WN - 2*li/Ip*Ipx;
lig =            - 2*li/Ip*Ipg;

% Derivatives of qA
TyAg = gAg.*(rA.*L.fPg.'+L.fTg.'./rA); % TyAg: Axis current (jphi.dR.dZ) contribution from each basis function
detH = dr2FA.*dz2FA-drzFA.^2;
trH  = dr2FA+dz2FA;
cqA  = mu0*rA.^2.*rBt.*sqrt(detH)./(abs(trH)*L.dsx);
D    = cqA.*(TyAg*ag);
qA   = (rBt.^2+(2e-7*L.idsx*IgAg.*L.fTg.')*ag)./D;
%
% Writing qA = N/D, the derivatives of qA can be computed using:
%   dqA_dx = dN_dx/D - qA/D*dD_dx = dN_dx/D - qA*dlnD_dx
% - the expression for N is:
%   N = rBt.^2+(2e-7*L.idsx*IgAg.*L.fTg.')*ag
% - the expression for D is:
%   D = cqA*TyAg*ag = mu0*rBt/L.dsx*sqrt(detH)/abs(trH)*gAg.*(rA.^3*L.fPg+rA*L.fTg).'*ag
%                   = mu0*rBt/L.dsx*sqrt(detH)/abs(trH)*D3;
%   The derivatives of lnD can be computed using:
%     dlnD_dx = 0.5*ddetH_dx/detH - dtrH_dx/trH + dD3_dx/D3
%   The expression for D3 is:
%     D3 = gAg.*(rA.^3*L.fPg+rA*L.fTg).'*ag
%        = gAg.*D1.'*ag
%
N1   = 2e-7*L.idsx*(ag.*L.fTg).';
D1   = rA.^3.*L.fPg.'+rA.*L.fTg.';
D2   = gAg.*(3*rA.^2*L.fPg.'+L.fTg.');
D3   = (gAg.*D1)*ag;
qAx  = reshape(sum(N1.*dIgAgdFx,2),nA,L.nx)./D - qA.*(0.5*ddetHdFx./detH-dtrHdFx./trH+reshape(sum(ag.'.*(dgAgdFx.*D1+reshape(full(drAdFx),nA,1,L.nx).*D2),2),nA,L.nx)./D3);
qAg  = (2e-7*L.idsx*IgAg.*L.fTg.' - qA.*cqA.*TyAg)./D;

% Derivatives of Wk
Wkx = ((ag.*L.fPg).' *dITpgdFx)*1.5;
Wkg = (     L.fPg .'.* ITpg   )*1.5;

% Assembly
%   Derivatives of Iy and Fx
if xHasIy
  dIydx = spdiags(L.xscal(L.ind.ixGS),L.ind.ixGS(1)-1,L.ny,L.nN); % [diag(L.xscal(L.ind.ixGS)),zeros(L.ny,L.ng+L.ne)]
  dFxdx = [L.Mxy ,zeros(L.nx ,L.ng),L.Mxe].*L.xscal.';
else
  dIydx = [-L.dlst./L.rhsf,zeros(L.ny,L.ng),-L.Tye].*L.xscal.';
  dFxdx = spdiags(L.xscal(L.ind.ixGS),L.ind.ixGS(1)-1,L.nx,L.nN); % [diag(L.xscal(L.ind.ixGS)),zeros(L.nx,L.ng+L.ne)]
end
% Quantities depending on Fx only
dFAdx  = dFAdFx*dFxdx;
dFBdx  = dFBdFx*dFxdx;
drAdx  = drAdFx*dFxdx;
dzAdx  = dzAdFx*dFxdx;
% Interpolated fluxes and fields
if L.nn
  [dFndx(:,:),dBrndx(:,:),dBzndx(:,:)] = L.P.infct(L.G.rx,L.G.zx,reshape(full(dFxdx),L.nzx,L.nrx,L.nN),L.P.rn,L.P.zn,L.inM);
else
  dFndx  = zeros(L.nn,L.nN);
  dBrndx = zeros(L.nn,L.nN);
  dBzndx = zeros(L.nn,L.nN);
end
% Quantities depending on Fx and ag
dagdx = spdiags(L.xscal(L.ind.ixg),L.ind.ixg(1)-1,L.ng,L.nN); % [zeros(L.ng,nGS),diag(L.xscal(L.ind.ig)),zeros(L.ng,L.ne)]
dFtdx  = Ftx *dFxdx + Ftg *dagdx;
dIpdx  = Ipx *dFxdx + Ipg *dagdx;
dbpdx  = bpx *dFxdx + bpg *dagdx;
dbpDdx = bpDx*dFxdx + bpDg*dagdx;
dlidx  = lix *dFxdx + lig *dagdx;
dqAdx  = qAx *dFxdx + qAg *dagdx;
dWkdx  = Wkx *dFxdx + Wkg *dagdx;
end
