%MEQC MEQ ancillary data
% L = MEQC(P,G,'P1',...) returns a structure with MEQ ancillary data. P
% contains the configuration parameters as obtained by LIUP<TOK> or
% FBTP<TOK>. G contains the geometry as obtained by LIUG<TOK> or FBTG<TOK>.
% It defines the mesh data by calling MEQMESH and the Poisson solver data
% by calling GSZRC, plus .Oly  Limiter domain. See also
% LIUP,LIUG,FBTP,FBTG,MEQMESH,GSZRC.
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
function L = meqc(P,G,varargin)
 
 L.P = P; L.G = G;
  
 %% Mesh related
 L = meqmesh(L,G.rx,G.zx,G.Mbc,varargin{:});

 %% Cubic spline interpolation
 if P.icsint
   assert(P.ilim == 3,'P.ilim must be 3 if P.icsint is true');
   L.nthreadscs = min(P.nthreadscs,maxNumCompThreads); % Do not use more threads than MATLAB has allocated
   [L.Mr,L.taur] = csdec(G.rx,G.rx,'a');
   [L.Mz,L.tauz] = csdec(G.zx,G.zx,'a');
 end

 %% Domains
 if isnan(P.ndom)
  % set maximum number of axes and domains based on idoublet
  if P.idoublet
   L.nD = 3;
  else
   L.nD = 1;
  end
 else
  assert(~P.idoublet || P.ndom>1,'Must select ndom>1 if idoublet==true')
  L.nD = P.ndom;
 end
 
 %% Solving Poisson equation
 [L.cx,L.cq,L.cr,L.cs,L.ci,L.co] = gszrc(L.ry,L.drx,L.dzx,L.nry,L.nzy);

 %% DLST operator and GS RHS factor
 % Needed for Newton-GS and all-nl-Fx
 L.dlst = dlstc(L.nzx,L.nrx,L.drx,L.dzx,L.rry);
 L.rhsf = (2.0 * L.dzx * L.idrx * pi * mu0) .* L.rry(:);
 
 %% Limiter domain
 assert(all(G.rl<=G.rx(end)) && all(G.zl<=G.zx(end)) && all(G.rl>=G.rx(1)) && all(G.zl>=G.zx(1)),'Limiter outside computational mesh')
 L.Oly = inpolygon(repmat(reshape(L.ry,1,[]),L.nzy,1),repmat(reshape(L.zy,[],1),1,L.nry),G.rl,G.zl);
 
 %% Interpolate flux on limiter
 [L.kxl,L.clx]   = bintc(G.rl ,G.zl ,G.rx,G.zx);
 if P.ilim == 2
   [L.kxlh,L.clhx] = bintc(G.rlh,G.zlh,G.rx,G.zx); % points between rl,zl for interpolation
 elseif P.ilim == 3
   assert(P.icsint>0,'P.icsint must be true if P.ilim==3');
   if isfield(G,'ppl')
     % limiter piecewise polynomial coefficients are present, convert to bsp format
     [L.Ml,L.taul] = pp2bsp(G.ppl);
   else
     % compute spline representation of limiter
     [M,L.taul] = csdec((0:G.nl)/G.nl,(0:G.nl)/G.nl,'p');
     L.Ml = M*[G.rl,G.zl;G.rl(1),G.zl(1)];
   end
   L.taul = reshape(L.taul,[],1);
 end

 %% Critical-point search domain == L.Oly plus neighbours or whole grid when P.lasy is true
 L.Oasx = false(L.nzx,L.nrx);
 L.Oasx(2:end-1,2:end-1) = L.Oly | P.lasy;
 L.Oasx(1:end-2,2:end-1) = L.Oasx(1:end-2,2:end-1) | L.Oly | P.lasy;
 L.Oasx(3:end  ,2:end-1) = L.Oasx(3:end  ,2:end-1) | L.Oly | P.lasy;
 L.Oasx(2:end-1,1:end-2) = L.Oasx(2:end-1,1:end-2) | L.Oly | P.lasy;
 L.Oasx(2:end-1,3:end  ) = L.Oasx(2:end-1,3:end  ) | L.Oly | P.lasy;
 
 %% meqpdom parameters
 % FN: marker value for non-extremum limiter points
 % dimw: working space for extrema/saddle points
 L.FN = -Inf; L.dimw = 200; % defaults for meqpdom
 
 %% theta-rho mesh
 % rho mesh - pq=rho(psiN);
 if ~isempty(P.pq)
  % take user-defined pq
  assert(numel(P.pq)>1,'pq must be an array')
  L.pq = [reshape(P.pq(P.pq > 0 & P.pq < 1),1,[]) 1];
 else
  % build array of npq points
  L.pq = [linspace(P.pql,sqrt(2)-sqrt(1-P.pqu^2),P.npq-1) sqrt(2)];
  k = L.pq>sqrt(0.5); 
  L.pq(k)= sqrt(1-(sqrt(2)-L.pq(k)).^2);
 end
 L.pQ = [0,L.pq];
 L.npq = numel(L.pq);
 L.pinit = 0.8;
 L.nQ = L.npq+1;
 % theta mesh
 L.noq = P.noq;
 L.doq = 2*pi/L.noq; L.oq = L.doq*(0:L.noq-1)';
 L.fq = 1-L.pq.^2; % normalized psiN with 0 at LCFS and 1 on axis
 L.crq  = -cos(L.oq);
 L.czq  =  sin(L.oq);
 L.cdrq = L.crq*L.idrx;
 L.cdzq = L.czq*L.idzx;
 L.nQ = L.npq+1;
 
 L.nS = numel(P.raS);
 L.nR = numel(P.iqR);
 assert(any(L.crq==1),'need noq to be even to have an outboard theta direction for computing r/a')
 
 msgid = 'MEQ:ParameterError';

 %% Fixed base function parameters
 % Handle default cases
 if ~isempty(P.bfct)
  % bf name
  bfname = func2str(P.bfct);

  % bf related checks
  bfp = bfpck(P.bfct,P.bfp,L.nD);

  % BFFBT is time-dependent
  isbffbt = strcmp(bfname,'bffbt'); % Only support specifying bffbt directly
  if isbffbt
   % handle bfp multi-time case
   bfp1 = bfp(:,1);
  else
   bfp1 = bfp;
  end

  % Consolidate bfp
  switch bfname
   case 'bfgenD'
    bfp1 = bfpgenD(bfp1,L.nD);
   case 'bfdoublet'
    bfp1 = bfpdoublet(bfp1);
  end

  % Use generic multidomain bfgenD if needed
  [~,~,TDg] = P.bfct(0,bfp1);
  if size(TDg,1) == 1 && L.nD>1
   L.bfct = @bfgenD;
   bfp1 = repmat({P.bfct,bfp1,zeros(L.nD,1)},L.nD,1);
   for iD = 1:L.nD
    bfp1{iD,3}(iD) = 1;
   end
   bfp1 = bfpgenD(bfp1,L.nD); % consolidate into a structure
  else
   L.bfct = P.bfct;
  end

  % Get ng/nq/nc
  [L.fPg,L.fTg,L.TDg] = L.bfct(0,bfp1);
  assert(~any(L.fPg.*L.fTg),msgid,'One basis function cannot be assigned to both p'' and TT'', all elements of fPg.*fTg must be 0');
  assert(size(L.TDg,1) == L.nD,msgid,'bfct=''%s'' can support only %d domains but L.nD=%d',bfname,size(L.TDg,1),L.nD);
  L.ng = numel(L.fPg); % #base functions
  o = ones(L.nD,1); z = zeros(L.nD,1);
  L.nq = size(L.bfct(6,bfp1,[],o,z,o,o,1),1); % #regularisations
  L.nc = size(L.bfct(7,bfp1,[],o,z      ),1); % #inequality constraints

  % Restore time-dependent parameters
  if isbffbt && L.nD>1 % Multi-domain with bfct=@bffbt
   bfp_ = bfp;
   nt = size(bfp_,2);
   bfp  = repmat(bfp1,1,nt);
   for it = 1:nt
    for ibf = 1:bfp1.nbf
     bfp(it).bfp{ibf} = bfp_(:,it);
    end
   end
  elseif ~isbffbt % Time independent parameters
    bfp = bfp1;
  end
  L.bfp = bfp;

 else
  L.fPg = []; L.fTg = []; L.TDg = ones(L.nD,0); L.ng = 0; L.nq = 0; L.nc = 0;
 end
 assert(L.ng > 0,'MEQ requires that the number of basis functions be strictly positive');
 
 %% linear interpolation indices for q95
 rho95 = sqrt(0.95);
 ii = find(L.pQ>=rho95,1,'first');
 if ~isempty(ii) && ii>2
  L.i95  = [ii-1,ii];
  L.c95  = 1-[rho95-L.pQ(ii-1), L.pQ(ii)-rho95]/diff(L.pQ(L.i95));
 else
  error(msgid,'can not compute q95 for this pq mesh')
 end
 
 %% Matrix for preparing interpolation
 if ~isempty(P.infct)
  % e.g. infct = qi5p.m becomes cfun=qi5pc.m
  cfun = str2func([strrep(func2str(P.infct),'mex',''),'c']);
  L.inM = cfun(P.inp,L.drx,L.dzx);
  L.nn = numel(P.rn);
 else
  L.nn = 0;
 end
 
 %% FS integrals
 if P.iterq
  % pre-calculate metrics for flux surface integrals
  % 1. rho-derivative
  [L.M1q,taup] = csdec(L.pQ,L.pQ,'w'); % cubic spline with dy/dp(0)=0 and not-a-knot bc at 1
  L.M1q = bspsum(taup,L.M1q,L.pq,1)./L.pq(:); % dy/dp/p
  %    The next line removes the need to have a point at pq=0 by assuming y(0)=0
  L.M1q = L.M1q(:,2:end).'; % y*L.M1q = dy/dp/p assuming y(0)=dy/dp(0)=0
  % 2. theta-derivative
  [L.M2q,tauo] = csdec([L.oq;2*pi],[L.oq;2*pi],'p'); % cubic spline with periodic bc
  L.M2q = bspsum(tauo,L.M2q,L.oq,1) / 2; % dy/do/2
  %    The next 2 lines remove the need to have a point at oq=2*pi by assuming y(2*pi)=y(0)
  L.M2q(:,1) = L.M2q(:,1) + L.M2q(:,end);
  L.M2q = L.M2q(:,1:end-1); % L.M2q*y = dy/do/2 assuming y 2pi periodic
  % 3. rho-derivative without dy/dp(0)=0 assumption
  [L.M3q,taup] = csdec(L.pQ,L.pQ,'a'); % cubic spline not-a-knot bc on both sides
  L.M3q = bspsum(taup,L.M3q,L.pq,1)./L.pq(:); % dy/dp/p
  L.M3q = L.M3q(:,2:end).'; % assuming y(0)=0
 end
 
 %% Gaps
 if G.nW
  L.erW  = (G.rW-G.rx(1))*L.idrx;
  L.ezW  = (G.zW-G.zx(1))*L.idzx;
  L.crW  = -cos(G.oW);
  L.czW  =  sin(G.oW);
  L.cdrW = L.crW*L.idrx;
  L.cdzW = L.czW*L.idzx;
  % Flux interpolation at origin of gaps (similar to limiter interpolation)
  [L.kxW,L.cWx] = bintc(G.rW,G.zW,G.rx,G.zx);
 end
 L.nW = G.nW*P.nFW;

 %% Conductors
 L.ne  = G.na + G.nu;

 %% Edge condition
 L.Mbe = [G.Mxa(~L.lxy,:),G.Mxu(~L.lxy,:)];

 %% Treatment of external currents inside the computational grid
 switch P.gsxe
  case 1 % Plasma external current in computational grid, Iy(kyW) = TWe*Ie(kew)
   rw = [G.rw ; G.rv]; % windings from coils and vessel
   zw = [G.zw ; G.zv];
   
   % windings within computational domain
   kw = find(rw > G.rx(1) & rw < G.rx(end) & ...
             zw > G.zx(1) & zw < G.zx(end));
   ky = sub2ind([L.nzy L.nry],...
     min(max(round((zw(kw)-L.zy(1))/L.dzx)+1,1),L.nzy),...
     min(max(round((rw(kw)-L.ry(1))/L.drx)+1,1),L.nry));
   assert(all(~L.Oly(ky)),'Some external current filaments are within the limiter domain')
   
   T = sparse(ky,kw,ones(size(ky)),L.ny,numel(rw)) * blkdiag(G.Twa, G.Tvu);
   % These are only used in liutsim now
   L.kew = find(any(T,1));
   L.kyw = find(any(T,2));
   L.Tww = T(L.kyw,L.kew);

   % Prepare matrix of external currents in computational grid
   L.Tye = sparse(T);
  case 2 % Use mutual for currents in computational grid
   % Prepare matrix of external currents in computational grid
   Mxie = [G.Mxia,G.Mxiu];
   Iyie = -(L.dlst*Mxie)./L.rhsf;
   Tye  = Iyie;
   % Discard Tye entries smaller than input tolerance (relative to each Ie)
   Tye(abs(Tye) < P.gsxetol*max(abs(Tye),[],1)) = 0;
   % Make it a sparse matrix to speed up Tye*Ie when there are few non-zero entries
   if nnz(Tye)/L.ny/L.ne<0.1, Tye = sparse(Tye); end
   L.Tye = Tye;
  case 3 % use green functions to solve the Poisson equation
   L.Mxe = [G.Mxa,G.Mxu];
   L.Mxy = G.Mxx(:,L.lxy(:));

   % Prepare matrix of external currents in computational grid
   Iyie = -(L.dlst*L.Mxe)./L.rhsf;
   L.Tye = Iyie;
 end
 
 %% Build Lackner's trick matrix representation
 assert(ismember(P.ilackner, [0, 1, 2]), 'Choice of ilackner is not supported');
 switch P.ilackner
   case 0
     L.Mby = G.Mby;
   case 2
     Mby = zeros(L.nx - L.ny, L.ny);
     for i=1:L.ny
       ei = zeros(L.ny, 1); ei(i) = 1.0;
       ei = reshape(ei, L.nzy, L.nry);
       Fb = L.Tbc*nfdbmex(gszrmex(zeros(2*(L.nzx+L.nrx-2),1),ei,L.cx,L.cq,L.cr,L.cs,L.ci,L.co,0));
       Mby(:, i) = Fb;
     end
     L.Mby = Mby;
 end

 %% Approximations
 L.smalldia  = false;

 %% Scaling
 % extra scaling factors added to improve global cost function landscape
 extra_ag0_scale = 10;
 extra_Iy0_scale = 20;
 % Nominal value for Ip
 rip = min(G.rl);
 rop = max(G.rl);
 Rl = 0.5*(rop+rip); % Major radius of limiter contour
 al = 0.5*(rop-rip); % Minor radius of limiter contour
 % Formula for Ip at fixed q95 from Sauter et al., 2016, FED
 % Ip[MA] = 4.1*a^2/R*BT/q95*f(shaping) with f of order 1
 % So we fix q95~4 and f=1 and compute the corresponding Ip
 L.Ip0 = 1e6*al^2/Rl*P.b0;

 % Nominal value for Fx
 % Flux difference created at Rl+al by filament at Rl carrying Ip0
 L.Fx0 = (greenem('self',Rl,0,[L.drx;L.dzx]) - greenem('mut',Rl,0,Rl+al,0))*L.Ip0;

 % Nominal value for Iy
 L.Iy0 = extra_Iy0_scale*L.Ip0/L.ny; % per point on y grid

 % Nominal value for ag
 % scale ag by typical value to get integrated Ip
 ri = G.rx(1); ro = G.rx(end); zl = G.zx(1); zu = G.zx(end);
 Fx = L.Fx0*(L.rrx-ri).*(L.rrx-ro).*(L.zzx-zl).*(L.zzx-zu).*16/(ro-ri)^2/(zu-zl)^2; % Fx0/0 at the center/boundary
 FA = L.Fx0*ones(L.nD,1); FB = zeros(L.nD,1);
 L.ag0 = ones(L.ng,1);
 for iD = 1:L.nD
   % Since here we set the plasma domain to the whole computational domain,
   % we don't have to differentiate basis functions that apply to multiple
   % domains
   [~,TpDg,~] = L.bfct(1,bfp1,Fx,FA,FB,repmat(int8(iD),L.nzy,L.nry),L.ry,L.iry);
   mask = logical(TpDg(iD,:)); % Only update scale of bfs that are non-zero
   L.ag0(mask) = abs(extra_ag0_scale*L.Ip0./TpDg(iD,mask));
 end
 
 % Nominal value for test function integrals over plasma domain for 1D CDEs
 L.intG0 = ones(L.ng, 1);
 extra_F0_scale = 0.3 / L.nD; % for a more flat current distributed on multiple domains
 FA_ = FA * extra_F0_scale;
 FB_ = FB * extra_F0_scale;
 g = L.bfct(91, bfp1, (1 - L.pQ.^2) * FA_(1), FA_, FB_);
 L.intG0 = 2 * pi * al^2 * abs(trapz(L.pQ, L.pQ .* g', 2));

 % Nominal value for Iu
 if G.nu
   L.Iu0 = (L.Ip0/G.nv)./max(abs(G.Tvu),[],1)';
 else
   L.Iu0 = zeros(0,1);
 end

 % Nominal value for Vu
 if isfield(G,'Ru') % Otherwise Vu probably not needed
   L.Vu0 = G.Ru.*L.Iu0;
 end

 % Nominal value for Ia
 % use limit if it is defined
 Ialim = max(abs(G.Iamin),abs(G.Iamax));
 iinf = isinf(Ialim);
 L.Ia0 = Ialim(:);
 % otherwise use current to generate poloidal field equal to 1/20 of B0 at
 % grid center
 if any(iinf)
   [Br0a,Bz0a] = greenem({'br','bz'},(ri+ro)/2,(zl+zu)/2,G.rw,G.zw,G.Twa);
   Bpa = sqrt(Br0a.^2+Bz0a.^2);
   Bp0 = P.b0/20;
   L.Ia0(iinf) = Bp0./Bpa;
 end

 % Nominal value for Va
 if isfield(G,'Ra') % Otherwise Va probably not needed
   % use limit if it is defined
   Valim = max(abs(G.Vamin),abs(G.Vamax));
   iinf = isinf(Ialim);
   L.Va0 = Valim(:);
   L.Va0(iinf) = G.Ra(iinf).*L.Ia0(iinf);
 end

 %% Txy/Txb
 % sparse matrices providing the embeddings and projections of the
 % boundary grid points, the Y-grid grid points and the X-grid grid
 % points are precomputed
 [L.Txy, L.Txb] = get_DOF_transfer_identities(L);

 %% Post-processing
 L.raN = 1; % default value for rational surface finder
 L.liurtemu = false; % Flag for when liut is used to emulate Simulink/rt runs
 % Additional output arguments
 L.argoutc = meqargoutc(P.argout);
 
end

function [Txy, Txb] = get_DOF_transfer_identities(L)
% GET_DOF_TRANSFER_IDENTITIES Computes sparse identity matrices to transfer 
% boundary grid points and inner grid points to their respective indices
% function [Txy, Txb] = get_DOF_transfer_identities(L)
% Arguments:
%   L:  struct, common L structure having all the L-structure stuff in it
% returns:
%   Txy: Sparse Matrix, (L.nx, L.ny)-matrix transferring a y grid vector
%        to the same on the x grid with zeros on the boundary (canonical
%        embedding of y grid in x grid)
%   Txb: Sparse Matrix, (canonical embedding of boundary points to x grid)
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% get inner and boundary grid indices
indices_inner = find(L.lxy);
indices_bdry = find(~L.lxy);

% define nb as it is not saved in L by default
nb = L.nx - L.ny;

% build the sparse matrices with just ones
Txy = sparse(indices_inner, 1:L.ny, ones(L.ny, 1), L.nx, L.ny);
Txb = sparse(indices_bdry, 1:nb, ones(nb, 1), L.nx, nb);
end
