function [Fc,Brc,Bzc,dnc,Fx,Brx,Bzx,dnx] = meqbdfield(L,sIp,rc,zc)
% MEQBDFIELD: Breakdown null field calculator
%
% Inputs:
%  L structure with P structure containing fields:
%   * rnull, znull: location of nulls
%   * onull: angle of null [rad] (see definition below)
%   * gBpnull: Gradient of (quadrupole) null field d|Bp|/dr
%   * nullmethod: method to compute null field (currently only 'fit' is supported).
%  sIp: sign of Ip, so that for quadrupole field with onull=0, HFS vertical
%       field will have the correct sign for Ip radial force balance.
%  rc,zc [optional]: control point locations on which to return target field.
%
%  Null angle definition (for sIp=1):
%    onull = 0: quadrupole with dBz/dz < 0 so HFS field has correct sign for radial force balance
%    onull > 0: following right-hand side rule in (R,phi,Z) coordinate system, 
%               rotates flux map clockwise in (R,Z) plane
%
% Outputs:
%  * Fc, Brc, Bzc: Poloidal flux, Br and Bz at rc,zc control points
%  * dnc: distance between each control point and the closest null
%  * Fx, Brx, Bzx, dnx: values on full x grid
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if nargout <= 4
  results_on_xgrid = false;
else
  results_on_xgrid = true; % compute results on x grid
end

switch L.P.nullmethod
  case 'fit'
    if results_on_xgrid
      [Fc,Brc,Bzc,Fx,Brx,Bzx] = bd_field_fit(L,sIp,rc,zc);
    else
      [Fc,Brc,Bzc] = bd_field_fit(L,sIp,rc,zc);
    end
  otherwise
    error('undefined bdfield method %s',L.P.bdfield)
end

rn = L.P.rnull;
zn = L.P.znull;

% Function to get distance to closest null
fun_d = @(rr,zz) min(sqrt((rr(:)-rn').^2+(zz(:)-zn').^2),[],2);
dnc  = reshape(fun_d(rc,zc),size(rc));

if results_on_xgrid
  dnx = reshape(fun_d(L.rrx(:),L.zzx(:)),L.nzx,L.nrx);
end

end

function [Fc,Brc,Bzc,Fx,Brx,Bzx] = bd_field_fit(L,sIp,rc,zc)
% Find null field by least-squares fit of circuit currents
% to obtain desired fields and field gradients at control points.

% desired fields at nulls
[Brn,Bzn,Bzrn,Brrn,Brrrn,Bzrrn] = null_field_parameters(L.P,sIp);

G = L.G;

rn = L.P.rnull;
zn = L.P.znull;
assert(iscolumn(rn),'rn must be a column vector')
assert(numel(rn)==numel(zn),'rn,zn must have same size')

[Brna,Bzna,Brrna,Bzrna] = greenem({'br','bz','dr1br','dr1bz'},rn,zn,G.rw,G.zw,G.Twa);

% second derivatives of field by finite differences
dd = 0.01; % distance for FD

[Brr1,Brz1] = greenem({'dr1br','dr1bz'},rn+dd,zn,G.rw,G.zw,G.Twa);
[Brr2,Brz2] = greenem({'dr1br','dr1bz'},rn-dd,zn,G.rw,G.zw,G.Twa);

Brrrna  = 1/(2*dd)*(Brr1 - Brr2);
Bzrrna  = 1/(2*dd)*(Brz1 - Brz2);

% LSQ fit for x=Ia:  min ||A*x - b||
nu = 1e-12; % regularization to keep Ia small

A = [Brna;
  Bzna;
  Brrna;
  Bzrna;
  Brrrna;
  Bzrrna;
  nu*eye(G.na)];

b = [Brn;
  Bzn;
  Brrn;
  Bzrn;
  Brrrn;
  Bzrrn;
  zeros(G.na,1)];

Ia = A\b;

[Mca,Brca,Bzca] = greenem({'mut','br','bz'},rc,zc,G.rw,G.zw,G.Twa);
[nc,nr] = size(rc); % reshape to same size as rc
Fc  = reshape( Mca*Ia,nc,nr);
Brc = reshape(Brca*Ia,nc,nr);
Bzc = reshape(Bzca*Ia,nc,nr);


if nargout>3
  % evaluate on x grid
  Fx = reshape(G.Mxa*Ia,L.nzx,L.nrx);
  Brx = reshape(G.Brxa*Ia,L.nzx,L.nrx);
  Bzx = reshape(G.Bzxa*Ia,L.nzx,L.nrx);
end
end

function [Br,Bz,Bzr,Brr,Brrr,Bzrr] = null_field_parameters(P,sIp)
% returns field and field gradients at null position
% given null parameters in P

thnull = P.onull; % null angle

% Null orientation and strength
nnull = size(P.rnull,1); % number of nulls
if nnull == 2
  vv = [1;-1]; % up-down mirrored
  oo = [1;1];
elseif nnull==1
  oo = 1;
  vv = 1; % just one
else
  error('can''t handle this amount of nulls')
end

Br = zeros(nnull,1); Bz = zeros(nnull,1); % no field at null
% field gradient at null, taking into account rotation and field gradient
Bzr  = oo.*cos(oo*2*thnull).*(sIp*P.gBpnull);
Brr  = vv.*sin(oo*2*thnull).*(sIp*P.gBpnull);
Brrr = zeros(nnull,1);
Bzrr = zeros(nnull,1);

end