function [bb] = meqbbox(rH,zH,rl,zl,idoublet)
% MEQBBOX returns bounding box information for points rH,zH
%
%   bb = meqbbox(rH,zH,rl,zl,idoublet)
% 
% Uses rl,zl to find approximate bounding box in case only 1 or 2 
% vertically/horizontally aligned points are given are given
%
% Returns bounding box points bb=[x1,x2,y1,y2] with x2>x1, y2>y1
% For doublet cases, bb has multiple rows, one for each domain
%
% see also BBOXMEX
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

%% Bounding rectangle
if isempty(rH) % no control points for this slice
  bb = nan(1,4);
elseif ~idoublet
  bb  =  [min(rH) max(rH) min(zH) max(zH) ];
else
  % assume two regions above and below mean of zH
  mzH = mean(zH);
  bb  = [
    [min(rH) max(rH) min(zH(zH>mzH)) max(zH(zH>mzH))];
    [min(rH) max(rH) min(zH(zH<mzH)) max(zH(zH<mzH))];
    ];
end

% Check for cases where the bbox is just a line or point. In this case use
% the limiter to guess the appropriate bbox size
for iD = 1:size(bb,1)
  if abs(bb(iD,3)-bb(iD,4)) < 10*eps && abs(bb(iD,1)-bb(iD,2)) < 10*eps % point
    rh = bb(iD,1); zh = bb(iD,3); % r,z of center
    dr = rh-rl; dz = zh-zl; % limiter distances
    d = min(sqrt(dr.^2 + dz.^2)); % minimum distance to limiter
    wh = d; hh = d;
    bb(iD,:) = [rh-wh/2 rh+wh/2 zh-hh/2,zh+hh/2]; % new bbox
  elseif abs(bb(iD,1)-bb(iD,2)) < 10*eps % points are on a vertical line
    hh = bb(iD,4)-bb(iD,3);
    wh = hh; % use height also as width
    rh = (bb(iD,1)+bb(iD,2))*0.5; % r coordinate
    bb(iD,1:2) = [rh-wh/2,rh+wh/2]; % new bbox
  elseif abs(bb(iD,3)-bb(iD,4)) < 10*eps % points are on a horizontal line
    wh = bb(iD,2)-bb(iD,1);
    hh = wh; % use width also as height
    zh = (bb(iD,3)+bb(iD,4))*0.5; % z coordinate
    bb(iD,3:4) = [zh-hh/2,zh+hh/2]; % new bbox
  end
end

end