function [ind,nN,dimN,xscal,resscal,LX2x,isEvolutive,nuN,nrD] = meqalgoNLc(L,algoNL,circuit)
% MEQALGONLC Ancillary data for Grad-Shafranov non-linear state description
%
% Inputs:
%   - L 
%   - algoNL: string describing plama state model ('picard','all-nl','Newton-GS')
%   - circuit: boolean, indicates whether circuit equations are included or not
%
% Outputs:
%   - ind : structure containing indices of different quantities for
%           the states (ix*), the residuals (ir*), the inputs (iu*)
%   - nN : number of non-linear states
%   - dimN : cell array of strings describing each state
%   - xscal : array of scales for the states (xSI = x.*xscal)
%   - resscal : array of scales for the residuals (resSI = res.*resscal)
%   - LX2x : function handle to extract the state from an LX or LY structure
%   - isEvolutive : boolean, indicates whether the non-linear model includes dynamic equations
%   - nuN : number of inputs for the non-linear model
%   - nrD : number of residuals corresponding to dynamic equations
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% Default for FGS or fgeFpicard
nxa = 0;
nxu = 0;

if strcmpi(algoNL,'picard')
  assert(L.nD==1,'Doublets not supported yet for Picard method');
  assert(L.np==0,'CDE not supported yet for Picard method');

  % plasma parametrized by only 2 numbers: axis locations
  [ind.ixGS,ind.ixg        ,ind.ixa,ind.ixu   ] = n2k(2,L.ng     ,nxa,nxu); % state
  [ind.irGS,ind.irC,ind.irp,ind.ira,ind.iru,nN] = n2k(2,L.ng,L.np,nxa,nxu); % residuals

  % State names
  dimN = [{'FR';'FZ'};...
          cellstr(num2str((1:L.ng)','ag_%03d'))];

  % Residual scaling
  resscal = ones(nN,1);
  % Typical value of FR,FZ in fgeFpicard
  % FR ~ h * r0 / 2*r0 = h/2
  % FZ ~ h * r0
  % where h is the typical value of the flux Hessian
  % at the axis. using tr(H) = -2*pi*mu0*rA*jphiA (Moret 2015 paper eq.88)
  % assuming H = diag([h,h]), we get tr(H) = 2h and
  % 2*h ~ -2*pi*mu*rA*jphiA
  h = pi*4e-7*pi*L.P.r0*L.idsx*L.Iy0; % typical value of H assuming jphiA = Iy0/dsx
  resscal(ind.irGS) = 1./[h/2;h*L.P.r0];
  resscal(ind.irC) = 1; % residuals are already scaled by meqagcon

  % State scaling
  xscal = ones(nN,1);
  if L.P.xscal
    xscal(ind.ixGS) = L.P.r0;
    xscal(ind.ixg) = L.ag0;
  end

  % State from L,LX
  LX2x = @(LX) [LX.rA;LX.zA;LX.ag]./xscal;
else
  switch lower(algoNL)
    case 'all-nl'
      nGS = L.ny;
      xscalGS = L.Iy0;
      rscalGS = 1./L.Iy0;
      varGS = 'Iy';
    case 'all-nl-fx'
      nGS = L.nx;
      xscalGS = L.Fx0;
      rscalGS = 1./L.Fx0;
      varGS = 'Fx';
    case 'newton-gs'
      nGS = L.nx;
      xscalGS = L.Fx0;
      rscalGS = ones(nGS,1);
      rscalGS(~L.lxy) = 1./L.Fx0;                       % res(~L.lxy) = Fb - Fb(Iy,Ie)
      rscalGS( L.lxy) = 1./abs(2*pi*mu0*L.P.r0.*L.Iy0); % res( L.lxy) = L.rhsf*(Iy-Iy(Fx,ag))
      varGS = 'Fx';
    otherwise
      error('Unknown algoNL=%s',algoNL);
  end

  % Number of external currents in the state vector
  if circuit
    % FGE
    nxa = L.G.na;
    nxu = L.G.nu;
  end

  % State: [GS;ag[;Ia;Iu]]
  [ind.ixGS,ind.ixg        ,ind.ixa,ind.ixu,nN] = n2k(nGS,L.ng     ,nxa,nxu);
  % Residual: [resGS;co[;CDE;circuit]]
  [ind.irGS,ind.irC,ind.irp,ind.ira,ind.iru   ] = n2k(nGS,L.nC,L.np,nxa,nxu);
  % State names
  dimN = [cellstr(num2str((1:nGS )',[varGS,'_%04d']));...
          cellstr(num2str((1:L.ng)',     'ag_%03d' ));...
          strcat('Ia_',L.G.dima(1:nxa));...
          strcat('Iu_',L.G.dimu(1:nxu))];

  % Residual scaling
  resscal = ones(nN,1);
  resscal(ind.irGS) = rscalGS;
  resscal(ind.irC) = 1; % residuals are already scaled by meqagcon
  resscal(ind.irp) = 1; % residuals are already scaled by meqcde
  if circuit
    resscal(ind.ira) = 1./L.Va0;
    resscal(ind.iru) = 1./L.Vu0;
  end

  % State scaling
  xscal = ones(nN,1);
  if L.P.xscal
    xscal(ind.ixGS) = xscalGS;
    xscal(ind.ixg) = L.ag0;
    if circuit
      xscal(ind.ixa) = L.Ia0;
      xscal(ind.ixu) = L.Iu0;
    end
  end

  % State from L,LX
  LX2x = @(LX) [LX.(varGS)(:);LX.ag;LX.Ia(1:nxa);LX.Iu(1:nxu)]./xscal;
end

isEvolutive = circuit;
    
% Inputs
if circuit
  % u = [Va;Co;IniD]
  [ind.iua,ind.iuC,ind.iuni,nuN] = n2k(L.G.na,L.nC,L.nD);
else
  % u = [Ia;Iu;Co]
  [ind.iua,ind.iuu,ind.iuC,nuN] = n2k(L.G.na,L.G.nu,L.nC);
end

% Static and dynamic equations
nrD = L.np+nxa+nxu;
[ind.irS,ind.irD] = n2k(nN-nrD,nrD);

end
