function S = meqagconfun(h)
% MEQAGCONFUN container for ag constraint functions
% Function signature:
% [res,dresdF0,dresdF1,dresdag,dresdFx,dresdTpDg, ...
%  dresdITpDg,dresdrA,dresddr2FA,dresddz2FA,dresddrzFA,dresdCo] = ...
% myconstraint(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg,iD)
%
% A previous version of meqagconfun returned A and b such that res = A*ag-b
% this can be recovered by computing A=dresdag and b = dresdag*ag-res.
%
% For nargout >= 2, all Jacobians of the residual with respect to the
% inputs are also returned. (If one of the Jacobians is zero, 0 is sometimes returned
% regardless of the correct size of the Jacobian).
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

fun   = localfunctions;
names = cellfun(@func2str,fun,'UniformOutput',false);
S = cell2struct(fun,names);
if nargin==0, return; end

if ischar(h) && isequal(h,'help')
  help(mfilename);
  fprintf('\nCatalog of available %s constraints\n',mfilename)
  for ii=1:numel(fun)
    myfun = names{ii};
    fprintf('%s:\n',myfun)
    help(sprintf('%s>%s',mfilename,myfun));
  end
end
end

function [res,dresdF0,dresdF1,dresdag,dresdFx,dresdTpDg,dresdITpDg, ...
          dresdrA,dresddr2FA,dresddz2FA,dresddrzFA,dresdCo] = ...
  ag(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg,ii)
% Directly specify ag = LX.ag

scal = 1./L.xscal(L.ind.ixg(ii));
res = scal*(ag(ii)-LX.ag(ii));

if nargout >= 2
  dresdF0 = 0; dresdF1 = 0; dresdFx = 0; dresdTpDg = 0; dresdITpDg = 0;
  dresdrA = 0; dresddr2FA = 0; dresddz2FA = 0; dresddrzFA = 0;
  
  A = zeros(1,L.ng);
  A(:,ii) = scal;
  dresdag = A;
  dresdCo = -scal;
end
end

function [res,dresdF0,dresdF1,dresdag,dresdFx,dresdTpDg,dresdITpDg, ...
          dresdrA,dresddr2FA,dresddz2FA,dresddrzFA,dresdCo] = ...
  li(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg,iD)
% constrain plasma li to LX.li (LIUQE li definition with fixed r0)
if numel(iD)~=1, error('only one domain supported for li constraint'); end
if L.nD == 1, li = LX.li;
else,         li = LX.liD(iD);
end
[Wp,~,~] = vizrmex(Fx,int8(Opy==iD),L.ry,L.iry,LX.rBt,L.drx,L.dzx);
Ip = TpDg(iD,:)*ag;
WN = 1e-7*pi*L.P.r0*Ip^2;

WN0 = 1e-7*pi*L.P.r0*L.Ip0.^2; % scaling factor to get terms ~1
% li = Wp/WN;
scal = 1./WN0;
res = scal*(Wp - li*WN);

if nargout >= 2
  dresdF0 = 0; dresdF1 = 0; dresdITpDg = 0;
  dresdrA = 0; dresddr2FA = 0; dresddz2FA = 0; dresddrzFA = 0;
  
  % gives the Wp derivative
  dWpdFx = vizrJac(Fx,int8(Opy==iD),L.iry,L.drx,L.dzx);
  
  dWNdag = 1e-7*pi*L.P.r0*Ip * TpDg(iD,:) * 2.0;
  dWNdTpDg = zeros(L.nD,L.ng);
  dWNdTpDg(iD,:) = 1e-7*pi*L.P.r0*Ip * ag.' * 2.0;
  
  dresdag   = -scal * li * dWNdag;
  dresdTpDg = -scal * li * dWNdTpDg(:).';
  dresdFx   =  scal * dWpdFx(:).';
  dresdCo   = -scal * WN;
end
end

function [res,dresdF0,dresdF1,dresdag,dresdFx,dresdTpDg,dresdITpDg, ...
          dresdrA,dresddr2FA,dresddz2FA,dresddrzFA,dresdCo] = ...
  Ip(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg,iD)
% Impose Ip = IpD(iD)
if numel(iD)~=1, error('only one domain supported for Ip constraint'); end
if L.nD == 1, Ip = LX.Ip;
else,         Ip = LX.IpD(iD);
end

scal = 1./L.Ip0;
A = scal*TpDg(iD,:);
res = A * ag - scal*Ip;

if nargout >= 2
  dresdF0 = 0; dresdF1 = 0; dresdFx = 0; dresdITpDg = 0;
  dresdrA = 0; dresddr2FA = 0; dresddz2FA = 0; dresddrzFA = 0;

  dIpdTpDg = zeros(L.nD,L.ng);
  dIpdTpDg(iD,:) = ag;
  
  dresdTpDg = scal * dIpdTpDg(:).';
  dresdag = A;
  dresdCo = -scal;
end
end

function [res,dresdF0,dresdF1,dresdag,dresdFx,dresdTpDg,dresdITpDg, ...
          dresdrA,dresddr2FA,dresddz2FA,dresddrzFA,dresdCo] = ...
  Wk(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg,iD)
% Impose WkD(iD) (kinetic energy per domain) or total Wk
if numel(iD)~=1, error('only one domain supported for Wk constraint'); end
if L.nD == 1, Wk = LX.Wk;
else,         Wk = LX.WkD(iD);
end
fPg = L.fPg;

WN0 = 1e-7*pi*L.P.r0*L.Ip0.^2; % scaling factor to get terms ~1
scal = 1/WN0;

% Wk = 1.5*ITpDg(iD,:).*fPg.'*ag;
A = scal*1.5*ITpDg(iD,:).*fPg.';

res = A * ag - scal*Wk;

if nargout >= 2
  dresdF0 = 0; dresdF1 = 0; dresdFx = 0; dresdTpDg = 0;
  dresdrA = 0; dresddr2FA = 0; dresddz2FA = 0; dresddrzFA = 0;
  
  dWkdITpDg = zeros(L.nD,L.ng);
  dWkdITpDg(iD,:) = 1.5*(fPg .* ag).';
  
  dresdag = A;
  dresdITpDg = scal*dWkdITpDg(:).';

  dresdCo = -scal;
end
end

function [res,dresdF0,dresdF1,dresdag,dresdFx,dresdTpDg,dresdITpDg, ...
          dresdrA,dresddr2FA,dresddz2FA,dresddrzFA,dresdCo] = ...
  bp(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg,iD)
% Impose bpD(iD) = LX.bpD(iD) (beta poloidal per domain) or total bp
% Uses LIUQE bp equation involving r0
if numel(iD)~=1, error('only one domain supported for bp constraint'); end
if L.nD == 1, bp = LX.bp;
else,         bp = LX.bpD(iD);
end
fPg = L.fPg;

Ip = TpDg(iD,:)*ag;
WN = 1e-7*pi*L.P.r0*Ip^2;

WN0 = 1e-7*pi*L.P.r0*L.Ip0.^2; % scaling factor to get terms ~1
scal = 1/WN0;

A = scal*ITpDg(iD,:).*fPg.';

res = A * ag - scal*bp*WN;

if nargout >= 2
  dresdF0 = 0; dresdF1 = 0; dresdFx = 0;
  dresdrA = 0; dresddr2FA = 0; dresddz2FA = 0; dresddrzFA = 0;
  
  dWNdag = 1e-7*pi*L.P.r0*Ip * TpDg(iD,:) * 2.0;
  dWNdTpDg = zeros(L.nD,L.ng);
  dWNdTpDg(iD,:) = 1e-7*pi*L.P.r0*Ip * ag.' * 2.0;

  dWkdITpDg = zeros(L.nD,L.ng);
  dWkdITpDg(iD,:) = (fPg .* ag).';
  
  dresdag = A - scal * bp * dWNdag;
  dresdTpDg = - scal * bp * dWNdTpDg(:).';
  dresdITpDg = scal * dWkdITpDg(:).';
  dresdCo = - scal * WN;
end
end

function [res,dresdF0,dresdF1,dresdag,dresdFx,dresdTpDg,dresdITpDg, ...
          dresdrA,dresddr2FA,dresddz2FA,dresddrzFA,dresdCo] = ...
  bt(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg,iD)
% Impose bt = LX.bt (toroidal beta toroidal)
if numel(iD)~=1, error('only one domain supported for bt constraint'); end
if L.nD == 1, bt = LX.bt;
else,         bt = LX.btD(iD);
end
fPg = L.fPg;

[~,Ft0,~] = vizrmex(Fx,int8(Opy==iD),L.ry,L.iry,LX.rBt,L.drx,L.dzx);
Wt0 = 2.5e6*LX.rBt*Ft0;

% Wk = 1.5*ITpDg(iD,:).*fPg.'*ag;
% scal = 100*mu0/pi/(B0^2*R0^2*Sx)
scal = 1e2./(2.5e6*L.P.r0*L.P.b0*(L.P.b0*L.P.r0*(L.nx*L.dsx)));
A = scal*ITpDg(iD,:).*fPg.';
res = A * ag - scal*bt*Wt0;

if nargout >= 2
  dresdF0 = 0; dresdF1 = 0; dresdFx = 0; dresdTpDg = 0;
  dresdrA = 0; dresddr2FA = 0; dresddz2FA = 0; dresddrzFA = 0;

  dWkdITpDg = zeros(L.nD,L.ng);
  dWkdITpDg(iD,:) = (fPg .* ag).';

  dresdag = A;
  dresdITpDg = scal * dWkdITpDg(:).';
  dresdCo = - scal * Wt0;
end
end

function [res,dresdF0,dresdF1,dresdag,dresdFx,dresdTpDg,dresdITpDg, ...
          dresdrA,dresddr2FA,dresddz2FA,dresddrzFA,dresdCo] = ...
  qA(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg,iD)
% Impose qA = LX.qA
% With qA calculated from axis quadratic expansion, as LIUQE paper eq.88
% with small diamagnetic approximation
nA = numel(rA);

if numel(iD)~=1 || iD>nA
  error('can impose qA constraint only on one axis domain')
end
qA = LX.qA(iD);
fPg = L.fPg;
fTg = L.fTg;
[gAg,IgAg] = L.bfct(5,L.bfp,[],F0,F1);

IgAg = 2e-7*L.idsx*IgAg;

% TyAg: Axis current (jphi.dR.dZ) contribution from each basis function [1,ng]
TyAg = gAg(iD,:).*(rA(iD)*fPg+fTg/rA(iD)).'; 

detH_sqrt = sqrt(dr2FA(iD)*dz2FA(iD)-drzFA(iD)^2);
trH = dr2FA(iD)+dz2FA(iD);
gA  = detH_sqrt/abs(trH);

cqA = 4e-7*pi*rA(iD)^2*LX.rBt*gA/L.dsx;

% res = rBt^2 - (A1+qA*A2)*ag so a natural scaling is (r0*b0)^2
scal = 1./(L.P.r0*L.P.b0).^2;
A = scal*((IgAg(iD,:).*fTg.') - qA*cqA*TyAg);
b = -scal*LX.rBt.^2;

res = A * ag - b;

if nargout >= 2
  dresdFx = 0; dresdTpDg = 0; dresdITpDg = 0;

  dresdrA = zeros(1,nA);
  dresddr2FA = dresdrA;
  dresddz2FA = dresdrA;
  dresddrzFA = dresdrA;
  
  dTyAgdrA = gAg(iD,:) .* (fPg - fTg / (rA(iD)^2)).';
  dcqAdrA = 2.0 * 4e-7*pi*rA(iD)*LX.rBt*gA/L.dsx;
  dresdrA(iD) = -scal*qA*(cqA*dTyAgdrA + dcqAdrA*TyAg) * ag;
  
  dgAddr2FA = 0.5 * dz2FA(iD) / (detH_sqrt * abs(trH)) - gA / trH;
  dgAddz2FA = 0.5 * dr2FA(iD) / (detH_sqrt * abs(trH)) - gA / trH;
  dgAddrzFA =      -drzFA(iD) / (detH_sqrt * abs(trH));
  
  gA_factor = -scal*qA*(TyAg * ag) * 4e-7*pi*rA(iD)^2*LX.rBt/L.dsx;
  dresddr2FA(iD) = gA_factor * dgAddr2FA;
  dresddz2FA(iD) = gA_factor * dgAddz2FA;
  dresddrzFA(iD) = gA_factor * dgAddrzFA;
  
  [dgAgdF0, dgAgdF1, dIgAgdF0, dIgAgdF1] = L.bfct(15,L.bfp,[],F0,F1);
  
  dTyAgdF0 = dgAgdF0(iD,:,:) .* (rA(iD)*fPg+fTg/rA(iD)).'; % size: [1,ng,nD]
  dTyAgdF1 = dgAgdF1(iD,:,:) .* (rA(iD)*fPg+fTg/rA(iD)).';
  dIgAgdF0 = 2e-7*L.idsx * dIgAgdF0(iD,:,:);
  dIgAgdF1 = 2e-7*L.idsx * dIgAgdF1(iD,:,:);
  
  dresdag = A;
  dresdF0 = scal*reshape(sum((dIgAgdF0.*fTg.' - qA * cqA * dTyAgdF0) .* ag.',2),1,L.nD);
  dresdF1 = scal*reshape(sum((dIgAgdF1.*fTg.' - qA * cqA * dTyAgdF1) .* ag.',2),1,L.nD);
  dresdCo = -scal*cqA*(TyAg*ag);
end
end

function [res,dresdF0,dresdF1,dresdag,dresdFx,dresdTpDg,dresdITpDg, ...
          dresdrA,dresddr2FA,dresddz2FA,dresddrzFA,dresdCo] = ...
  fbtlegacy(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg,iD)
% Directly specify ag(1) == ag(2)

ig = find(L.TDg(iD,:));
if numel(ig) ~= 3
  error('fbtlegacy requires 3 basis functions per domain');
end
res = ag(ig(1)) - ag(ig(2));

if nargout >= 2
  dresdF0 = 0; dresdF1 = 0; dresdFx = 0; dresdTpDg = 0; dresdITpDg = 0;
  dresdrA = 0; dresddr2FA = 0; dresddz2FA = 0; dresddrzFA = 0;
  
  dresdag = zeros(1,L.ng);
  dresdag(ig) = [1,-1,0];

  dresdCo = 0;
end
end
