function [dIypdFy,dIypdF0,dIypdF1,dIypdFx] = meqIyJac(L,ag,mask,dTygdFy,dTygdF0,dTygdF1,dF0dFx,dF1dFx)
% MEQIYJAC Assembles analytical gradient of Iy with respect to Fy,F0,F1 and total Fx
%
%  [dIypdFy,dIypdF0,dIypdF1,dIypdFx] = meqIyJac(L,ag,mask,dTygdFy,dTygdF0,dTygdF1[,dF0dFx,dF1dFx])
%
% Iyp depends on the local value of Fx (dIypdFy) and on the value of F0 and
% F1 (dIypdF0/dIypdF1). So computing the variation of Iyp for a given
% variation of Fx can be faster using dIypd[Fy,F0,F1] than computing the
% full dIypdFx jacobian.
%
% An important note is that dIypdFx is row-sparse meaning rows corresponding
% to points outside of the plasma domain are zero.
% - When L.P.icsint=false, the non-zero rows have only a few non-zero elements so
% we return a sparse matrix of size [ny,nx]
% - When L.P.icsint=true, the non-zero rows are dense so we return a dense matrix
% excluding the zero rows so the size is [nyp,nx] where nyp is the number of points
% within the plasma domain
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

dIypdFy = dTygdFy(mask,:)*ag; nyp = size(dIypdFy,1);
dIypdF0 = reshape(sum(dTygdF0(mask,:,:).*ag.',2),nyp,L.nD);
dIypdF1 = reshape(sum(dTygdF1(mask,:,:).*ag.',2),nyp,L.nD);

if nargout>3
  % construct total Fx derivative
  if nargin<8
    error('Computing dIypdFx requires dF0dFx/dF1dFx');
  end
  if L.P.icsint
    % Use full matrix
    dIypdFx = zeros(nyp,L.nx);
    % dIydFy
    ii = (1:nyp).';
    jj = find(L.lxy); jj = jj(mask);
    dIypdFx(ii + (jj-1)*nyp) = dIypdFy;
    % dIyd[F0/F1]
    dIypdFx = dIypdFx + dIypdF0*dF0dFx + dIypdF1*dF1dFx;
  else
    % Use sparse matrix
    % dIydFy
    ii = find(mask);
    jj = find(L.lxy); jj = jj(mask);
    dIypdFx = sparse(ii,jj,dIypdFy,L.ny,L.nx,L.ny+12*L.nD*nyp); % 12: 6-point stencil for each of F0/F1
    % dIyd[F0/F1]
    iiD = repmat(ii(:),1,L.nD);
    jjD = repmat(1:L.nD,nyp,1);
    dIypdF0_ = sparse(iiD,jjD,dIypdF0,L.ny,L.nD);
    dIypdF1_ = sparse(iiD,jjD,dIypdF1,L.ny,L.nD);
    dIypdFx = dIypdFx + dIypdF0_*dF0dFx + dIypdF1_*dF1dFx;
  end
end
end
