%MEQFXJAC  Compute derivatives of MEQFX
%
%   [Mxy,Mxe] = meqFxjac(L)
%
% Computes the derivatives of MEQFX with respect to the plasma current
% distribution Mxy and to the external current Mxe.
%
% See also MEQFX
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
function [Mxy,Mxe] = meqFxjac(L)

% Short-circuit for gsxe=3
if L.P.gsxe == 3
  Mxy = L.Mxy;
  Mxe = L.Mxe;
  return
end

% Handle sizes
nr = L.nrx;
nz = L.nzx;

nr2 = nr-2;
nz2 = nz-2;

nb = 2*(nz+nr2);
ny = nr2*nz2;
ntot = nb+ny;

% Construct array of Poisson problems
Fb = zeros(nb     ,ntot);
Iy = zeros(nz2,nr2,ntot);
Fb(1:nb+1:nb^2) = 1;
Iy(nb*ny+(1:ny+1:ny^2)) = 1;

% Solve all problems at once
Fx = gszrmex(Fb,Iy,L.cx,L.cq,L.cr,L.cs,L.ci,L.co,0);

% Separate into independent Fb and Iy solutions
Fx = reshape(Fx,L.nx,ntot);
dgszrdFb = Fx(:,1:nb);
dgszrdIy = Fx(:,nb+1:end);

% Combine into Mxy ...
Mxy = dgszrdFb*(L.Tbc*nfdbmex(reshape(dgszrdIy,nz,nr,ny)))+dgszrdIy;

% ... and Mxe
Mxe = dgszrdFb*L.Mbe + dgszrdIy*L.Tye;

end