%LIUT  LIUQE reconstruction
% LY = LIUT(L,LX[,'PAR',VAL,...]) performs the LIUQE reconstuction using the
% parameters in L as obtained by LIUC and the measurements in LX as obtained by
% LIUX<TOK>, optionally replacing or adding parameters with specified values. It
% returns a structure LY with the reconstructed equilibria, with fields:
%
% LIUT(...,'rst',false,...) can be used to insert a custom initial guess.
% Note that this is only allowed when LX contains a single time
% slice.Values for IyD,Ie,dz should be also provided as name-value pairs.
% Values for aq,aW,z0,LYp can also be provided when desired. Combined with
% itert>0, one can then use LIUT as an iterator. 
%
% For a listing of generic LY output fields, see help meqt.m
%
% LIU-specific output fields:
% .Xt    DML synthetic measurement
% .dz    Vertical shift
% .dzg   Fitted vertical shift
% .chih  Fit residual for FE least-squares fit of currents
% .rst   Reset flag
% .cycle Computation time
%
% .chi   Root-mean-square of cost function residual
% .chie  Root-mean-square of constraint residual 
% .werr  Weighted cost function error
% .uerr  Unweighted cost function error
% .ID    Equation id for each cost function term
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

function LY = liut(L,LX,varargin)

%% Run on a single thread
% In some versions it is not possible to call maxNumCompThreads with
% an argument if MATLAB was started with the "-singleCompThread" option.
% In this case the call without any argument will return 1
if ~isempty(which('maxNumCompThreads')) && (maxNumCompThreads>1)
  nthreads = maxNumCompThreads(1);
  cleanupObj = onCleanup(@() maxNumCompThreads(nthreads));
end

%% Arguments
 Iy_init = true;  Ie_init = true; dz_init = true; 
 ag_init = true;  Co_init = true;
 aq_init = true;  aW_init = true; z0_init = true;
LYp_init = true; rst_init = true;

for k = 1:2:numel(varargin)
  switch varargin{k}
    % Initialise iterator state via arguments
    case 'Iy' , Iy  = varargin{k+1}; Iy_init = false;
    case 'Ie' , Ie  = varargin{k+1}; Ie_init = false;
    case 'dz' , dz  = varargin{k+1}; dz_init = false;
    case 'ag' , ag  = varargin{k+1}; ag_init = false;
    case 'Co' , Co  = varargin{k+1}; Co_init = false;
    case 'aq' , aq  = varargin{k+1}; aq_init = false;
    case 'aW' , aW  = varargin{k+1}; aW_init = false;
    case 'z0' , z0  = varargin{k+1}; z0_init = false;
    case 'LYp', LYp = varargin{k+1};LYp_init = false;
    case 'rst', rst = varargin{k+1};rst_init = false;
    % Additional output arguments
    case 'argout', L.argoutc = meqargoutc(varargin{k+1},L.argoutc);
    % Additional parameters
    otherwise, L.P.(varargin{k}) = varargin{k+1};
  end
end
 
% Allocate variables (for first slice) not provided as inputs
if  Iy_init, Iy  = zeros(L.nzy,L.nry,L.nD); end
if  Ie_init, Ie  = zeros(L.ne,1);  end
if  dz_init, dz  = zeros(L.ndz,1); end
if  ag_init, ag  = zeros(L.ng,1); end
if  Co_init, Co  = [];   end
if  aq_init, aq  = [];   end
if  aW_init, aW  = [];   end
if  z0_init, z0  = 0;    end
if LYp_init, LYp = [];   end
if rst_init, rst = true; end

%% Basic checks
assert(~L.P.slx,'Can not run liut.m with slx=1, call `liutsim.m` if you want to run Simulink version')
assert(rst || numel(LX.t) == 1,'Providing a custom initial condition can only work with a single LX slice')
if rst && (~Iy_init || ~Ie_init), warning('Iy or Ie passed but rst=true, FE fit will be run and passed Iy/Ie will be replaced'); end

%% ag/dz constraints (temporary ... or not)
if ~isfield(LX,'ag'), LX.ag = zeros(L.ng ,numel(LX.t)); end
if ~isfield(LX,'dz'), LX.dz = zeros(L.ndz,numel(LX.t)); end

%% QP solver setup
[qpsolve,qpopts] = meqqpsetup(L.P);

%% NL method
xHasIy = strcmp(L.P.algoNL,'all-nl');

% Scale
Uscale = abs([L.Ia0;L.Iu0;L.ag0;repmat(L.P.fdz,L.ndz,1)]);
nU = numel(Uscale);

%% Prepare for Picard iterations
idts = 0;
% dxdu = -Jx\Ju
[picard_Precs,dxdus] = fgeFJac_inverse_picard_approx(L,idts);
dxdus{1} = [dxdus{1},zeros(L.nN,L.nC)];
% Note: This is only dxdIe so far, x(Co) is non-linear
%       dxdCo needs to be computed at each iteration
dxdxps = {}; % un-needed

%% Time loop
ctol = 1e-6;      % Constraint tolerance for post-LSQ checks
nt = numel(LX.t);
msgtype = 'w';
niter = 0;
wsSQP = L.P.wsSQP;

if ~isfield(LX,'IpD')
  LX.IpD(1:L.nD,:) = LX.Ip/L.nD;
end

for kt = 1:nt
  tstart = tic;

  LXt = meqxk(LX,kt);

  LXt.tol = L.P.psichco;
  LXt.niter = L.P.itera;
  LXt.isaddl = L.P.isaddl;
  LXt.idoublet = L.P.idoublet;
  
  %% Measurements
  Ip = LXt.Ip;
  
  %% Get optimization problem
  [Raxc,Rauc,Razc,Ra0c,Rexc,Reuc,Rezc,Re0c,Rixc,Riuc,Rizc,Ri0c, ...
    IDs,ws,TusU,TdzsU,d1dt,d2dt] = liulsqinit(L,LXt);
  
  %% Initial guess using FE fit
  Jh = repmat(Ip/L.nh,L.nh,1);
  if rst
    % do FE fit
    Yd = Ra0c{1}(1:L.nd);
    Xe = [LXt.Ia; LXt.Iu]; % external currents
    [Ie,Jh,Zd] = lihlsqsolve(L,Yd,Xe,Jh,Ip);
    chih = sqrt(sum((Zd-Yd).^2)/L.nd); % chi of current fit
    Iy = reshape(L.Tyh*Jh,L.nzy,L.nry);
  else
    chih = NaN;
  end
  
  %% Assign x and u
  if kt==1 || rst

    % Initial x
    ag_ = ag;  Iy_ = Iy; Ie_ = Ie;
    % Get GS part of state
    if xHasIy % all-nl
      xGS_ = Iy_;
    else % all-nl-Fx, Newton-GS
      xGS_ = meqFx(L,Iy_,Ie_);
    end
    % Assign full state
    x = [xGS_(:);ag_]./L.xscal;
    
    % Initial u
    if Co_init % initial Co if not provided through init. 
      Co_ = LX2Co(L,LXt);
    else
      Co_ = Co; 
    end
    u = [Ie_;Co_];

    % Prepare meqopt inputs
    xs = {x};
    us = {u};
    zs = {dz};
    RaUt = zeros(size(Ra0c{1},1),nU);
    ReUt = zeros(size(Re0c{1},1),nU);
    RiUt = zeros(size(Ri0c{1},1),nU);
    dxs_ = {zeros(L.nN,1+L.nuN)};
    masku = true(L.nuN,1); % u=[Ia;Iu;Co]
    % Set Iy for Ip<Ipmin cases
    LXt.Iy = Iy;
    L.P.wsSQP = wsSQP;
  else
    LXt.Iy = zeros(L.nzy,L.nry);
    L.P.wsSQP = 0;
  end

  % Initialisation status
  s = [];
  if abs(LXt.Ip)<L.P.Ipmin
    s.msg = sprintf('Ip too low: %3.2e',Ip);
    s.msgid = 'LowIp';
  end

  % Set z0 (only used when zfct is liuz0)
  if isfield(L.P,'z0'), z0 = L.P.z0;
  elseif rst,           z0 = (L.Ipzh*Jh)/(L.Iph*Jh);end
  zfct = @(dzg) L.P.zfct(L.P,rst,z0,z0,dzg);
  % Set aq,aW
  LXt.aq = aq;
  LXt.aW = aW;

  init_response_matrix = false;

  %% NL iterations
  [xs,us,zs,LYs,dxs_,conv,dzg] = meqopt(L,LXt,...
    xs,us,zs,s,init_response_matrix,RaUt,ReUt,RiUt,...
    Raxc,Rauc,Razc,Ra0c,Rexc,Reuc,Rezc,Re0c,Rixc,Riuc,Rizc,Ri0c,...
    IDs,ws,TusU,TdzsU,d1dt,d2dt,...
    idts,picard_Precs,dxdus,dxdxps,dxs_,Uscale,masku,...
    msgtype,qpsolve,qpopts,ctol,zfct);

  rst = ~conv;
  
  %% Post-processing
  if conv || L.P.LYall || L.P.itert
    % single-slice is assumed
    LYt = LYs{1};
    dz = zs{1};

    % liu specific outputs
    % - override measurement fits with LIUQE versions which include dz term
    if L.P.stabz && any(dz)
      Vr = LYt.Xr;
      LYt.Ff = Vr(L.kdf);
      LYt.Bm = Vr(L.kdm);
    end
    % - DML synthetic measurement (already present from fgeF when idml=1)

    % - additional outputs
    LYt = meqlarg(LYt,dz,dzg,rst);
    % chi
    if L.P.itert == 0
      LYt = meqlarg(LYt,chih);
    else
      niter = LYt.niter+niter*(kt>1 && ~rst);
      LYt.niter = niter; % Cumulative number of iterations without restarting
    end
    LYt.cycle = toc(tstart)*1e6; % cycle time in us

    % Update aq, aW
    aq = LYt.aq; aW = LYt.aW;

    % Clean-up leftover special outputs from fgeF - keep Xt
    xtrafields = {     'dXtdx','dXtdu',...
                  'Xq','dXqdx','dXqdu',...
                  'Xc','dXcdx','dXcdu',...
                  'Xr','dXrddz'};
    LYt = rmfield(LYt,xtrafields(isfield(LYt,xtrafields)));

    if isempty(LYp)
      LYt = meqLYdot(LYt,LYt);
    else
      LYt = meqLYdot(LYt,LYp);
    end
    
    LY(kt) = LYt; %#ok<AGROW>
    LYp = LYt; % store previous
  end

  % Prepare FE fit for next iteration if itert=0
  rst = rst || L.P.itert == 0;
end % for kt

if exist('LY','var') % There is at least one converged time slice
  LY = meqlpack(LY);
elseif nargout > 0
  LY = struct([]);
end

end

