function [Ie,aj,Zd,st,id,msg] = liulsqsolve(L,rst,nB,Ie0,aj,Yr,Yk,Xc,Wrj,Wkj,Qcj,Ye,Xe)
% LIULSQSOLVE Solve LIU linear least squares problem
%
%   [Ie,Jh,Zd] = lihlsqsolve(L,Yd,Xe,Jh,sIp)
%
% Inputs:
%   L    LIH/LIU ancillary data structure
%   rst  Flag indicating whether LIU fit was reset, impacts aj selection
%   nB   Number of active domains, impacts aj selection
%   Ie0  Current estimate of [Ia;Iu]
%   aj   Current estimate of [ag;dz]
%   Yr   Vector of weighted measurements [Bm;Ff]
%   Yk   Vector of weighted measurements [Ip;Xt;Xq;aj]
%   Xc   Vector of inequality constraint RHS
%   Wrj  Response matrix for [Bm;Ff] as function of aj
%   Wkj  Response matrix for [Ip;Xt;Xq;aj] as function of aj
%   Qcj  Response matrix for inequality constraints Qcj*aj>=Xc
%   Ye   Vector of weighted measurements [Ia;Iu]
%   Xe   Vector of external currents measurements (used as initial guess)
%
% Outputs:
%   Ie   Fitted external currents [Ia;Iu]
%   aj   Fitted basis function amplitudes (ag) and vertical stabilization
%        parameters (dz)
%   Zd   Reconstructed values of the weighted measurements [Bm;Ff;Ia;Iu;Ip;Xt]
%   st   status flag (1 for success, 0 for failure)
%   id   error identifier when unsuccessful
%   msg  error message when unsuccessful
%
% This function solves the following linear problem in a least squares sense:
%   [Wre Wrj]   [Ie]   [Yr]
%   [Wee  0 ] * [aj] = [Ye]    with Qcj*aj>=Xc
%   [ 0  Wkj]          [Yk]
% Some Ie and aj can be exactly constrained:
%   - L.ke(ie) is false when Ie(ie) is exactly constrained. The
%   corresponding row in Wee is set to 0.
%   - L.kj(ij) is false when aj(ij) is exactly constrained. The
%   corresponding row in Wjj (part of Wkj) is set to 0.
% The inequality constraints are only active if L.P.ipm is true.
% If L.P.iters>0, an iterative solver is used where each iteration finds
% first the optimum for Ie and then for aj.
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

%% Init
% Status
st = 1;
id = '';
msg = '';
% Response matrices
ZWre = L.Wre;
ZWee = L.Wee;
ZWrj = Wrj;
ZWkj = Wkj;
ZQcj = Qcj;
% Measurements
ZYr = Yr;
ZYe = Ye;
ZYk = Yk;
ZXc = Xc;

% Rescale dz part
kjdz = L.ng+1:L.nj;
ZWrj(:,kjdz) = L.P.fdz*ZWrj(:,kjdz);
ZWkj(:,kjdz) = L.P.fdz*ZWkj(:,kjdz);
ZQcj(:,kjdz) = L.P.fdz*ZQcj(:,kjdz);
aj(kjdz) = aj(kjdz)/L.P.fdz;

%% Ie constraints
ke = L.ke;
Ie = Xe;
if L.nee > 0
  % Update measurements depending on Ie
  ZYr = ZYr - ZWre(:,~ke)*Ie(~ke);
  ZWre = ZWre(:,ke);
  ZWee = ZWee(:,ke);
end

%% aj constraints (so far only of the form aj=Xj with weights, no general linear constraints)
% aj for inactive domains are added to exact constraints
kj = L.kj;
if L.nje > 0 || (rst && L.P.itert == 0 && nB ~= 1) || nB < L.nD % L.nje = number of exact constraints
  aj(kj) = 0; % This also sets coefficients from unused domains to 0
  if (rst && L.P.itert == 0 && nB~=1), kj(L.ng+1:end) = false; end        % All dzs removed
  if (nB < L.nD),                      kj(~any(L.TDg(1:nB,:),1)) = false; % Remove ag for missing domains
                                       kj(L.ng+nB+1:end) = false; end     % Remove dz for missing domains
  % Adjust A and b for exact constraints
  ZYr = ZYr - ZWrj*aj;
  ZYk = ZYk - ZWkj*aj;
  ZXc = ZXc - ZQcj*aj;
  ZWrj = ZWrj(:,kj);
  ZWkj = ZWkj(:,kj);
  ZQcj = ZQcj(:,kj);
end
nja = sum(kj);

%% Fit sources to measurements
if L.P.iters
  % iterative solver
  if L.P.ipm
    [ZIe,Zaj,~] = ipmj(ZWrj,ZWkj,ZWre,L.Aer,L.Aee,ZYr,ZYe,ZYk,Ie0(ke),[],ZQcj,ZXc,[],[],1e-6,L.P.iters);
  else
    iZAjj = iatamex(ZWrj,ZWkj);
    ZAjr = iZAjj * ZWrj';
    [Zaj,ZIe] = bslvmex(ZAjr*ZYr+iZAjj*(ZWkj'*ZYk), L.Aer*ZYr+L.Aee*ZYe, Xe(ke), ZAjr*ZWre,L.Aer*ZWrj, L.P.iters);
  end
else
  % direct non-iterative solver
  A = [ZWre              ZWrj           ; ...
       ZWee              zeros(L.ne,nja); ...
       zeros(L.nk,L.nea) ZWkj           ];
  b = [ZYr; ...
       ZYe; ...
       ZYk];
  % if isfield(LX,'ro')
  %  AdG = [AdG ; L.wo.*[L.Moa L.Mos L.Moy*Tyg L.dzMoy*Iy(:)]];
  %  if isempty(LX.po)
  %   Yd = [Yd ; zeros(L.no,1)];
  %  else
  %   Yd = [Yd ; L.wo.*(FA + LX.po.^2*FBA)];
  %  end
  % end
  if L.P.robust
    a = robustfit(A,b,[],[],'off');
  elseif L.P.ipm
    [a,~,~,~,~,stat] = ipm(A'*A,-A'*b,[zeros(L.nc,L.nea),ZQcj],ZXc,[],[],[Ie0(ke);aj(kj)],[],[],[],2);
    if stat ~= 1
      st = 0; msg = 'IPM no convergence'; id = 'IPMFail';
    end
  else
    a = A\b;
  end
  ZIe = a(1:L.nea);
  Zaj = a(L.nea+1:end);
end
aj(kj) = Zaj;
Ie(ke) = ZIe;
% Rescale dz
aj(kjdz) = L.P.fdz*aj(kjdz);
% Synthetic measurements
Zd = [L.Wre*Ie+Wrj*aj ; L.Wee*Ie ; Wkj(L.kki,:)*aj];
end
