function [   Mnf,    Mnm,    Mna,    Mnu,    Mnp,...
           dvMnf,  dvMnm,  dvMna,  dvMnu,  dvMnp,...
          dv2Mnf, dv2Mnm, dv2Mna, dv2Mnu, dv2Mnp] = lih2qshapobs(L,rn,zn,vrn,vzn)
% LIH2QSHAPOBS get quadratic shape observer matrices using LIH matrices/methods.
%         
%[   Mnf,    Mnm,    Mna,    Mnu,    Mnp,...
%  dvMnf,  dvMnm,  dvMna,  dvMnu,  dvMnp,...
% dv2Mnf, dv2Mnm, dv2Mna, dv2Mnu, dv2Mnp] = lih2shapobs(L,rn,zn,vrn,vzn)
% 
% Outputs matrices to compute first and second derivatives along v of the
% flux at interpolation points. Distance of LCFS to interpolation point
% along the line v can the be estimated by solving
%     FB = Fn + dvFn * x + 0.5 * dv2Fn * x^2
% for x to give an accurate position error also in the presence of X points
%
% Inputs:
% . L structure from lih.m
% . (rn, zn) coordinates of interpolation points
% . (vrn, vzn) directions of required derivatives at interpolation points
% 
% Returns observer matrices such that:
% Fn    = [   Mnf,    Mnm,    Mna,    Mnu,    Mnp] * Xd;
% dvFn  = [ dvMnf,  dvMnm,  dvMna,  dvMnu,  dvMnp] * Xd;
% dv2Fn = [dv2Mnf, dv2Mnm, dv2Mna, dv2Mnu, dv2Mnp] * Xd;
% where Xd are the (unweighted) measurements: [Ff;Bm;Ia;Iu;Ip]
% 
% See also LIHT, LIHLSQSOLVE
% 
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% get Lih matrices to produce Ie and Jh
[Med,Mhd,kdu,kdp] = lih2obs_setup(L);

% generate mutual inductance matrices (and derivs) for interpolation points
[Mne, Mnh, dvMne, dvMnh, dv2Mne, dv2Mnh] = get_mut_mats(L,rn,zn,vrn,vzn);

% combine observer matrices with lih matrices
[   Mnf,    Mnm,    Mna,    Mnu,    Mnp] = dispatch(   Mne,    Mnh, Med, Mhd, L, kdu, kdp);
[ dvMnf,  dvMnm,  dvMna,  dvMnu,  dvMnp] = dispatch( dvMne,  dvMnh, Med, Mhd, L, kdu, kdp);
[dv2Mnf, dv2Mnm, dv2Mna, dv2Mnu, dv2Mnp] = dispatch(dv2Mne, dv2Mnh, Med, Mhd, L, kdu, kdp);
end


function [Anf, Anm, Ana, Anu, Anp] = dispatch(Ane, Anh, Med, Mhd, L, kdu, kdp)
Anf = Ane * Med(:,L.kdf) + Anh  * Mhd(:,L.kdf);
Anm = Ane * Med(:,L.kdm) + Anh  * Mhd(:,L.kdm);
Ana = Ane * Med(:,L.kda) + Anh  * Mhd(:,L.kda);
Anu = Ane * Med(:,  kdu) + Anh  * Mhd(:,  kdu);
Anp = Ane * Med(:,  kdp) + Anh  * Mhd(:,  kdp);
end


function [Mne, Mnh, dvMne, dvMnh, dv2Mne, dv2Mnh] = get_mut_mats(L,rn,zn,vrn,vzn)
% init
inM = qintc([],L.drx,L.dzx);
nn = numel(rn);

% influence of Ie on flux and derivatives computed using grid interpolation
% to correctly treat coils overlapping the grid
Fxh = meqFx(L, zeros(L.nzy, L.nry, L.ne), eye(L.ne));
[Mne,Brne,Bzne,~,Brzne,Bzrne,Bzzne] = qintmex(L.rx,L.zx,Fxh,rn,zn,inM);
% get flux derivatives from magnetic fields and reshape
Mne    = reshape(                     Mne,                  nn, L.ne);
drMne  = reshape( 2 * pi * rn(:) .*  Bzne,                  nn, L.ne);
dzMne  = reshape(-2 * pi * rn(:) .*  Brne,                  nn, L.ne);
drrMne = reshape( 2 * pi * rn(:) .* Bzrne + 2 * pi .* Bzne, nn, L.ne);
drzMne = reshape( 2 * pi * rn(:) .* Bzzne,                  nn, L.ne);
dzzMne = reshape(-2 * pi * rn(:) .* Brzne,                  nn, L.ne);
% derivatives along v
dvMne = drMne .* vrn + dzMne .* vzn;
dv2Mne = drrMne .* vrn .* vrn + 2 * drzMne .* vrn .* vzn + dzzMne .* vzn .* vzn;

% generate influence from Jh using grid interpolation to avoid closeness
% to singularities
Iyh = reshape(L.Tyh,L.nzy,L.nry,L.nh);
Fxh = meqFx(L, Iyh, zeros(L.ne, L.nh));

[Mnh,Brnh,Bznh,~,Brznh,Bzrnh,Bzznh] = qintmex(L.rx,L.zx,Fxh,rn,zn,inM);
% get flux derivatives from magnetic fields and reshape
Mnh    = reshape(                     Mnh,                  nn, L.nh);
drMnh  = reshape( 2 * pi * rn(:) .*  Bznh,                  nn, L.nh);
dzMnh  = reshape(-2 * pi * rn(:) .*  Brnh,                  nn, L.nh);
drrMnh = reshape( 2 * pi * rn(:) .* Bzrnh + 2 * pi .* Bznh, nn, L.nh);
drzMnh = reshape( 2 * pi * rn(:) .* Bzznh,                  nn, L.nh);
dzzMnh = reshape(-2 * pi * rn(:) .* Brznh,                  nn, L.nh);
% derivatives along v
dvMnh = drMnh .* vrn + dzMnh .* vzn;
dv2Mnh = drrMnh .* vrn .* vrn + 2 * drzMnh .* vrn .* vzn + dzzMnh .* vzn .* vzn;
end