function [Med,Mhd,kdu,kdp] = lih2obs_setup(L)
% LIH2OBS_SETUP generate LIH matrices for linear observers
% [Med,Mhd,kdu,kdp] = lih2obs_setup(L)
% Multiplying matrices Med, Mhd with measurements yields same results for
% Ie and Jh as running liht with iterh=0
%
% Arguments:
%   L: struct, Lih L struct
% returns:
%   Med: double(ne, nd), Matrix mapping Xd measurement vector to Ie
%   Mhd: double(nh, nd), Matrix mapping Xd measurement vector to Jh
%   kdu: int(nu),        Indices of Iu components if L.P.ivesm
%   kdu: int,            Index of Ip component if L.P.Ipmeas
% 
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

assert(L.P.iterh==0,'need iterh==0 for lih observers')

%% Ie constraints
Pdr = zeros(L.nd,L.nr); Pdr(L.kdr,:) = eye(L.nr);
Ped = zeros(L.ne,L.nd); Ped(:,L.kde) = eye(L.ne);

% Matrix such that Yd = MYX*Xd including exact constraint correction
Wd = diag([L.Wr;L.We;L.Wi]);
ke = L.ke;

MYX = Wd;
if L.nee
  % Correction for exactly constrained conductor currents
  MYX = MYX - Pdr*(L.Wre*diag(~ke))*Ped;
end

% matrix such that Jh = Mhd*Xd
Mhd = L.Ahd*MYX;
% matrix such that Ie(ke) = ZMed*Xd
ZMed = L.Aed*MYX;

% matrix such that Ie = Med*Xd
Med = zeros(L.ne,L.nd);
Med(ke,:) = ZMed;   % lsq solved Ie
Med(~ke,L.kde(~ke)) = eye(L.nee); % directly assigned Ie

% Ip special
if ~L.P.Ipmeas % no direct Ip measurement available
  % Plasma current estimator Ip = Ipm*Bm + Ipa*Ia + Ipu*Iu
  Tdd = eye(L.nd);
  Tdd(L.kdp,[L.kdm,L.kda,L.kdu]) = [L.G.Ipm,L.Ipa,L.P.ivesm*L.Ipu];
  Tdd(:,L.kdp) = []; % Remove explicit Ip measurement
  Med = Med*Tdd;
  Mhd = Mhd*Tdd;
  kdp = []; % Update index
else
  kdp = L.kdp;
end

if ~L.P.ivesm
  kdu = [];
else
  kdu = L.kdu;
end