/* [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved. */

/*
 * bfab.c contains the definiton of the bfab basis function set
 *
 * See also bfab.m bfef.c
 */

# include "meq.h"

/* List of function pointers to use when creating the bfct_params structure */
#define FUNPARAMS FLT_NAME(bfab_f) , FLT_NAME(bfab_fN) , FLT_NAME(bfab_fag) , FLT_NAME(bfab_fA) , \
                  FLT_NAME(bfab_df), FLT_NAME(bfab_dfN), FLT_NAME(bfab_dfag), FLT_NAME(bfab_dfA)

/*
 * The elementary functions to evaluate the necessary quantities for all
 * polynomial orders are shared with bfef
 */

/*
 * Next we define some functions to map all polynomial orders to each basis
 * function
 */

/* Function to dispatch all orders to 1...ng */
void FLT_NAME(bfab_dispatch)(int narray, FLT ** arrayin, FLT ** arrayout, void *params)
{
  int nP, nT;
  int i;

  /* Get number of basis functions for each class */
  nP = ((bfab_params *)params)->nP;
  nT = ((bfab_params *)params)->nT;
  /* Dispatch to output arrays, skipping first element from bfef */
  for (i=0; i<narray; i++) {
    (arrayin[i])++;
    memcpy(  arrayout[i]     , arrayin[i], nP*sizeof(FLT));
    memcpy(&(arrayout[i][nP]), arrayin[i], nT*sizeof(FLT));
  }
}

void FLT_NAME(bfab_fPg)(int nP, int nT, FLT fPg[], FLT fTg[])
{
  /* Use bfef version */
  FLT_NAME(bfef_fPg)(nP, nT, fPg, fTg);
}

/*
 * Next we define functions that will evaluate the full basis function set
 * These functions first call the elementary functions defined at the top
 * of this file and then dispatches their results based on (nP,nT) values.
 * These functions are the ones used in the generic bfct functions.
 */

void FLT_NAME(bfab_f)(void *params, FLT g[], FLT Ig[], FLT FxA, FLT FBA)
{
  FLT g_[4], Ig_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {g_,Ig_};
  FLT * arrayout[2] = {g ,Ig };

  /* Compute all orders */
  FLT_NAME(bfef_f_)(g_, Ig_, FxA, FBA);
  /* Dispatch to 1...ng */
  FLT_NAME(bfab_dispatch)(2, arrayin, arrayout, params);
}

void FLT_NAME(bfab_fN)(void *params, FLT gN[], FLT IgN[], FLT FN)
{
  FLT gN_[4], IgN_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {gN_,IgN_};
  FLT * arrayout[2] = {gN ,IgN };

  /* Compute all orders */
  FLT_NAME(bfef_fN_)(gN_, IgN_, FN);
  /* Dispatch to 1...ng */
  FLT_NAME(bfab_dispatch)(2, arrayin, arrayout, params);
}

void FLT_NAME(bfab_fA)(void *params, FLT g[], FLT Ig[], FLT FBA)
{
  FLT g_[4], Ig_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {g_,Ig_};
  FLT * arrayout[2] = {g ,Ig };

  /* Compute all orders */
  FLT_NAME(bfef_fA_)(g_, Ig_, FBA);
  /* Dispatch to 1...ng */
  FLT_NAME(bfab_dispatch)(2, arrayin, arrayout, params);
}

void FLT_NAME(bfab_fag)(void *params, FLT alphapg[], FLT alphag[], FLT FBA)
{
  FLT alphapg_[4], alphag_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {alphapg_,alphag_};
  FLT * arrayout[2] = {alphapg ,alphag };

  /* Compute all orders */
  FLT_NAME(bfef_fag_)(alphapg_, alphag_, FBA);
  /* Dispatch to 1...ng */
  FLT_NAME(bfab_dispatch)(2, arrayin, arrayout, params);
}

void FLT_NAME(bfab_df)(void *params, FLT  d1g[], FLT  d2g[],
                                     FLT d1Ig[], FLT d2Ig[],
                                     FLT FxA, FLT FBA)
{
  FLT d1g_[4], d2g_[4], d1Ig_[4], d2Ig_[4];
  /* 4 arrays to compute */
  FLT * arrayin [4] = {d1g_,d2g_,d1Ig_,d2Ig_};
  FLT * arrayout[4] = {d1g ,d2g ,d1Ig ,d2Ig };

  /* Compute all orders */
  FLT_NAME(bfef_df_)(d1g_, d2g_, d1Ig_, d2Ig_, FxA, FBA);
  /* Dispatch to 1...ng */
  FLT_NAME(bfab_dispatch)(4, arrayin, arrayout, params);
}

void FLT_NAME(bfab_dfN)(void *params, FLT dgN[], FLT dIgN[], FLT FN)
{
  FLT dgN_[4], dIgN_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {dgN_,dIgN_};
  FLT * arrayout[2] = {dgN ,dIgN };

  /* Compute all orders */
  FLT_NAME(bfef_dfN_)(dgN_, dIgN_, FN);
  /* Dispatch to 1...ng */
  FLT_NAME(bfab_dispatch)(2, arrayin, arrayout, params);
}

void FLT_NAME(bfab_dfA)(void *params, FLT dg[], FLT dIg[], FLT FBA)
{
  FLT dg_[4], dIg_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {dg_,dIg_};
  FLT * arrayout[2] = {dg ,dIg };

  /* Compute all orders */
  FLT_NAME(bfef_dfA_)(dg_, dIg_, FBA);
  /* Dispatch to 1...ng */
  FLT_NAME(bfab_dispatch)(2, arrayin, arrayout, params);
}

void FLT_NAME(bfab_dfag)(void *params, FLT dalphapg[], FLT dalphag[], FLT FBA)
{
  FLT dalphapg_[4], dalphag_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {dalphapg_,dalphag_};
  FLT * arrayout[2] = {dalphapg ,dalphag };

  /* Compute all orders */
  FLT_NAME(bfef_dfag_)(dalphapg_, dalphag_, FBA);
  /* Dispatch to 1...ng */
  FLT_NAME(bfab_dispatch)(2, arrayin, arrayout, params);
}

/*
 * These functions are the ones used in Simulink S-function blocks for each
 * mode and in MEX-files when a custom implementation is present or needed
 *  - modes 1-5, 8: the generic bfct functions are called
 *  - mode 6 (regularisation constraints) has a custom implementation
 *  - mode 7 (inequality constraints) has a custom implementation
 */

/* Transfer matrix and integrals over plasma domain */
void FLT_NAME(bfab1)(FLT *Tyg, FLT *Tpg, FLT *ITpg,
                     FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                     FLT *ry, FLT *iry, int nzy, int nry,
                     int nP, int nT, int err) {
  /* Parameters */
  FLT fPg[nP+nT], fTg[nP+nT];
  bfab_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, fPg, fTg, &fparams, FUNPARAMS};

  /* Compute fPg and fTg */
  FLT_NAME(bfab_fPg)(nP, nT, fPg, fTg);
  /* Call generic function */
  FLT_NAME(bfct1)(Tyg, Tpg, ITpg,
                 Fx, FA, FB, Opy,
                 ry, iry, nzy, nry, err,
                 &params);
}

/* Normalized basis functions */
void FLT_NAME(bfab2)(FLT *gQg, FLT *IgQg, FLT *FNQ, int nQ, int nP, int nT) {
  /* Parameters */
  bfab_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, NULL, NULL, &fparams, FUNPARAMS};

  /* Call generic function */
  FLT_NAME(bfct2)(gQg, IgQg, FNQ, nQ,
                  &params);
}

/* Coefficients to transform normalized basis functions into Pp/P/TTp/hqT */
void FLT_NAME(bfab3)(FLT *aPpg, FLT *aTTpg, FLT *aPg, FLT *ahqTg, FLT *ag, 
                     FLT FA, FLT FB, FLT ids, int nP, int nT) {
  /* Parameters */
  FLT fPg[nP+nT], fTg[nP+nT];
  bfab_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, fPg, fTg, &fparams, FUNPARAMS};

  /* Compute fPg and fTg */
  FLT_NAME(bfab_fPg)(nP, nT, fPg, fTg);
  /* Call generic function */
  FLT_NAME(bfct3)(aPpg, aTTpg, aPg, ahqTg, ag,
                  FA, FB, ids,
                  &params);
}

/* Vertical chord integrals */
void FLT_NAME(bfab4)(FLT *Tdg, FLT *Tgy, FLT *Fx, FLT FA, FLT FB, int8_t *Opy, int nzy, int nry,
                     unsigned int *kd, FLT *fd, int nd, int nP, int nT) {
  /* Parameters */
  bfab_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, NULL, NULL, &fparams, FUNPARAMS};

  /* Call generic function */
  FLT_NAME(bfct4)(Tdg, Tgy, Fx, FA, FB, Opy, nzy, nry,
                  kd, fd, nd,
                  &params);
}

/* Physical basis functions at magnetic axis */
void FLT_NAME(bfab5)(FLT *gA, FLT *IgA, FLT FA, FLT FB, int nP, int nT) {
  /* Parameters */
  bfab_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, NULL, NULL, &fparams, FUNPARAMS};

  /* Call generic function */
  FLT_NAME(bfct5)(gA, IgA, FA, FB,
                  &params);
}

/* Regularization constraints */
void FLT_NAME(bfab6)(FLT *Qqg, FLT *Xq, FLT FA, FLT FB, FLT rA, FLT irA, FLT ids, int nP, int nT, int nq) {
  int nPq = nP ? nP-1 : 0;
  /* int nTq = nT ? nT-1 : 0; */
  /* int nq = nPq + nTq; */
  FLT FBA = FB-FA, FBA2 = FBA*FBA, FBA3 = FBA2*FBA;
  /* Arrays init */
  memset(Qqg,0,nq*(nP+nT)*sizeof(FLT));
  memset(Xq ,0,nq        *sizeof(FLT));
  /* */
  switch (nP) {
    case 3: Qqg[     2*nq+    1] = 3.464101615137754*FBA3*rA*ids; /* sqrt(12) */
    case 2: Qqg[       nq      ] = 2.000000000000000*FBA2*rA*ids;
  }
  switch (nT) {
    case 3: Qqg[(nP+2)*nq+nPq+1] = 3.464101615137754*FBA3*irA*ids;
    case 2: Qqg[(nP+1)*nq+nPq  ] = 2.000000000000000*FBA2*irA*ids;
  }
}

/* Inequality constraints */
void FLT_NAME(bfab7)(FLT *Qcg, FLT *Xc, FLT FA, FLT FB,
                     int nP, int nT, int nc) {
  FLT FBA = FB-FA;
  /* Arrays init */
  memset(Qcg,0,nc*(nP+nT)*sizeof(FLT));
  memset(Xc ,0,nc        *sizeof(FLT));
  switch (nP) {
    case 3: Qcg[5] = FBA*FBA;
    case 2: Qcg[2] = Qcg[3] = FBA; Qcg[1] = 1.0;
    case 1: Qcg[0] = 3.0;
  }
}

/* Toroidal magnetic field amplitude */
void FLT_NAME(bfab8)(FLT *Bty, FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                     FLT *ag, FLT rBt, FLT ids, FLT *iry, int nzy, int nry,
                     int nP, int nT, int err) {
  /* Parameters */
  FLT fPg[nP+nT], fTg[nP+nT];
  bfab_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, fPg, fTg, &fparams, FUNPARAMS};

  /* Compute fPg and fTg */
  FLT_NAME(bfab_fPg)(nP, nT, fPg, fTg);
  /* Call generic function */
  FLT_NAME(bfct8)(Bty, Fx, FA, FB, Opy,
                  ag, rBt, ids, iry, nzy, nry, err,
                  &params);
}

/* Derivatives of transfer matrix and integrals over plasma domain */
void FLT_NAME(bfab11)(FLT  *dTygdFy, FLT  *dTygdFA, FLT  *dTygdFB,
                                     FLT *dITygdFA, FLT *dITygdFB,
                      FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                      FLT *ry, FLT *iry, int nzy, int nry,
                      int nP, int nT, int err) {
  /* Parameters */
  FLT fPg[nP+nT], fTg[nP+nT];
  bfef_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, fPg, fTg, &fparams, FUNPARAMS};

  /* Compute fPg and fTg */
  FLT_NAME(bfef_fPg)(nP, nT, fPg, fTg);
  /* Call generic function */
  FLT_NAME(bfct11)(dTygdFy, dTygdFA,  dTygdFB,
                           dITygdFA, dITygdFB,
                   Fx, FA, FB, Opy,
                   ry, iry, nzy, nry, err,
                   &params);
}

/* Derivatives of physical basis functions at magnetic axis */
void FLT_NAME(bfab15)(FLT *dgA1, FLT * dgA2, FLT *dIgA1, FLT *dIgA2,
                      FLT FA, FLT FB, int nP, int nT) {
  /* Parameters */
  bfef_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, NULL, NULL, &fparams, FUNPARAMS};

  /* Call generic function */
  FLT_NAME(bfct15)(dgA1, dgA2, dIgA1, dIgA2, FA, FB,
                   &params);
}

/* Derivatives of regularization constraints */
void FLT_NAME(bfab16)(FLT *dQqgdFA, FLT *dQqgdFB, FLT *dQqgdrA, FLT *dQqgdirA,
                      FLT FA, FLT FB, FLT rA, FLT irA, FLT ids, int nP, int nT, int nq) {
  int nPq = nP ? nP-1 : 0;
  /* int nTq = nT ? nT-1 : 0; */
  /* int nq = nPq + nTq; */
  FLT FBA = FB-FA, FBA2 = FBA*FBA, FBA3 = FBA2*FBA;
  /* Arrays init */
  memset(dQqgdFA,0,nq*(nP+nT)*sizeof(FLT));
  memset(dQqgdFB,0,nq*(nP+nT)*sizeof(FLT));
  memset(dQqgdrA,0,nq*(nP+nT)*sizeof(FLT));
  memset(dQqgdirA,0,nq*(nP+nT)*sizeof(FLT));
  /* */
  switch (nP) {
    case 3: dQqgdFA[     2*nq+    1] = -10.39230484541326 *FBA2*rA*ids; /* 3*sqrt(12) */
            dQqgdFB[     2*nq+    1] =  10.39230484541326 *FBA2*rA*ids; /* 3*sqrt(12) */
            dQqgdrA[     2*nq+    1] =   3.464101615137754*FBA3   *ids; /*   sqrt(12) */
    case 2: dQqgdFA[       nq      ] =  -4.000000000000000*FBA *rA*ids;
            dQqgdFB[       nq      ] =   4.000000000000000*FBA *rA*ids;
            dQqgdrA[       nq      ] =   2.000000000000000*FBA2   *ids;
  }
  switch (nT) {
    case 3: dQqgdFA[(nP+2)*nq+nPq+1] = -10.39230484541326 *FBA2*irA *ids; /* 3*sqrt(12) */
            dQqgdFB[(nP+2)*nq+nPq+1] =  10.39230484541326 *FBA2*irA *ids; /* 3*sqrt(12) */
            dQqgdirA[(nP+2)*nq+nPq+1] =   3.464101615137754*FBA3     *ids; /*   sqrt(12) */
    case 2: dQqgdFA[(nP+1)*nq+nPq  ] =  -4.000000000000000*FBA *irA *ids;
            dQqgdFB[(nP+1)*nq+nPq  ] =   4.000000000000000*FBA *irA *ids;
            dQqgdirA[(nP+1)*nq+nPq  ] =   2.000000000000000*FBA2     *ids;
  }
}

/* Derivatives of inequality constraints */
void FLT_NAME(bfab17)(FLT *dQcgdFA, FLT *dQcgdFB, FLT FA, FLT FB,
                      int nP, int nT, int nc) {
  FLT FBA = FB-FA;
  /* Arrays init */
  memset(dQcgdFA,0,nc*(nP+nT)*sizeof(FLT));
  memset(dQcgdFB,0,nc*(nP+nT)*sizeof(FLT));
  switch (nP) {
    case 3: dQcgdFA[5] = -FLTC(2.0)*FBA;
            dQcgdFB[5] =  FLTC(2.0)*FBA;
    case 2: dQcgdFA[2] = dQcgdFA[3] = -FLTC(1.0);
            dQcgdFB[2] = dQcgdFB[3] =  FLTC(1.0);
  }
}

