/* [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved. */

/*
 * bf3p.c contains the definiton of the bf3p basis function set
 *
 * The bf3p set corresponds to bfab[nP=1,nT=2] except if the boolean
 * parameter circ is true in which case it corresponds to bfab[nP=0,nT=1]
 *
 * It does not use the generic bfct functions to optimize the execution
 * time
 *
 * See also bfef.c bfct.c bfab.m
 */

# include "meq.h"

/* List of function pointers to use when creating the bfct_params structure */
#define FUNPARAMS FLT_NAME(bf3p_f) , FLT_NAME(bf3p_fN) , FLT_NAME(bf3p_fag) , FLT_NAME(bf3p_fA) , \
                  FLT_NAME(bf3p_df), FLT_NAME(bf3p_dfN), FLT_NAME(bf3p_dfag), FLT_NAME(bf3p_dfA)

/*
 * First we define some elementary functions to evaluate the necessary 
 * quantities for all polynomial orders. These will later be used when
 * mapped to each basis function
 */

/* Compute physical basis functions */
void FLT_NAME(bf3p_f)(void *params, FLT g[3], FLT Ig[3], FLT FxA, FLT FBA)
{
  meq_bool circ;
  
  circ = ((bf3p_params *) params)->circ;

   g[0] =  FxA - FBA ;                       /* Fx - FB                                       */
   g[1] =  g[0] ;                            /* Fx - FB                                       */
   g[2] =  g[0] * FxA;                       /* (Fx - FA) (Fx - FB)                           */
  Ig[0] =  0.5 * g[0] * g[0];                /* (Fx - FB)^2 / 2                               */
  Ig[1] =  Ig[0];                            /* (Fx - FB)^2 / 2                               */
  Ig[2] =  Ig[1] * (TWOITHREE * g[0] + FBA); /* (Fx - FB)^2 ((Fx - FB) 2 / 3 + (FB - FA)) / 2 */

  if (circ) {
    /* For low elongation geometry */
    g[0] = g[2] = Ig[0] = Ig[2] = 0.0;
  }
}

/* Compute normalized basis functions */
void FLT_NAME(bf3p_fN)(void *params, FLT gN[3], FLT IgN[3], FLT FN)
{
  meq_bool circ;
  
  circ = ((bf3p_params *) params)->circ;
  
   gN[0] = FN - 1.0;                         /*  FN - 1                   */
   gN[1] = gN[0];                            /*  FN - 1                   */
   gN[2] = gN[0] *  FN;                      /* (FN - 1) FN               */
  IgN[0] = 0.5 * gN[0] * gN[0];              /* (FN - 1)^2 / 2            */
  IgN[1] = IgN[0];                           /* (FN - 1)^2 / 2            */
  IgN[2] = IgN[0] * (TWOITHREE*FN + ITHREE); /* (FN - 1)^2 (2 FN + 1) / 6 */

  if (circ) {
    /* For low elongation geometry */
    gN[0] = gN[2] = IgN[0] = IgN[2] = 0.0;
  }
}

/* Compute physical basis functions at magnetic axis */
void FLT_NAME(bf3p_fA)(void *params, FLT g[3], FLT Ig[3], FLT  FBA)
{
  meq_bool circ;
  
  circ = ((bf3p_params *) params)->circ;
  
   g[0] = -FBA ;                 /*  FA - FB           */
   g[1] =  g[0];                 /*  FA - FB           */
   g[2] =  0.0;                  /*  0                 */
  Ig[0] =  0.5 * FBA * FBA;      /* (FA - FB)^2 / 2    */
  Ig[1] =  Ig[0];                /* (FA - FB)^2 / 2    */
  Ig[2] =  ITHREE * Ig[0] * FBA; /* (FA - FB)^3 / (-6) */

  if (circ) {
    /* For low elongation geometry */
    g[0] = g[2] = Ig[0] = Ig[2] = 0.0;
  }
}

/* Conversion factors from normalized to physical basis functions */
void FLT_NAME(bf3p_fag)(void *params, FLT alphapg[3], FLT alphag[3], FLT FBA)
{
  alphapg[0] =            FBA; /* (FB-FA)   */
  alphapg[1] = alphapg[0]    ; /* (FB-FA)   */
  alphapg[2] = alphapg[0]*FBA; /* (FB-FA)^2 */
   alphag[0] = alphapg[2]    ; /* (FB-FA)^2 */
   alphag[1] = alphapg[2]    ; /* (FB-FA)^2 */
   alphag[2] = alphapg[2]*FBA; /* (FB-FA)^3 */
}

/* Compute normalized basis functions */
void FLT_NAME(bf3p_dfN)(void *params, FLT dgN[3], FLT dIgN[3], FLT FN)
{
  meq_bool circ;
  
  circ = ((bf3p_params *) params)->circ;
  
   dgN[0] = 1.0;          /*  1            */
  dIgN[0] = FN - 1.0;     /* (FN - 1)      */
   dgN[1] = dgN[0];       /*  1            */
  dIgN[1] = dIgN[0];      /* (FN - 1)      */
   dgN[2] = dIgN[0] + FN; /*  2 FN - 1     */
  dIgN[2] = dIgN[0] * FN; /* (FN - 1) * FN */

  if (circ) {
    /* For low elongation geometry */
    dgN[0] = dgN[2] = dIgN[0] = dIgN[2] = 0.0;
  }
}

/* Compute derivatives of physical basis functions */
void FLT_NAME(bf3p_df)(void *params, FLT  dgdFxA[3], FLT  dgdFBA[3],
                                      FLT dIgdFxA[3], FLT dIgdFBA[3],
                                      FLT FxA, FLT FBA)
{
  meq_bool circ;
  
  circ = ((bf3p_params *) params)->circ;
  
   dgdFxA[0] =  1.0;                                              /* 1                                      */
  dIgdFxA[0] =  FxA - FBA;                                        /* Fx - FB                                */
   dgdFxA[1] =   dgdFxA[0];                                       /* 1                                      */
  dIgdFxA[1] =  dIgdFxA[0];                                       /* Fx - FB                                */
   dgdFxA[2] =  dIgdFxA[1] + FxA;                                 /* 2 Fx - FA - FB                         */
  dIgdFxA[2] =  dIgdFxA[1] * FxA;                                 /* (Fx - FB) (Fx - FA)                    */
   dgdFBA[0] = -1.0;                                              /* -1                                     */
  dIgdFBA[0] = -dIgdFxA[1];                                       /* FB - Fx                                */
   dgdFBA[1] =   dgdFBA[0];                                       /* 1                                      */
  dIgdFBA[1] =  dIgdFBA[0];                                       /* Fx - FB                                */
   dgdFBA[2] = -FxA;                                              /* FA - Fx                                */
  dIgdFBA[2] =  0.5 * dIgdFxA[1] * dIgdFBA[1] + dIgdFBA[1] * FBA; /* -0.5*(FB - Fx)^2 + (FB - Fx) (FB - FA) */

  if (circ) {
    /* For low elongation geometry */
    dgdFxA[0] = dgdFxA[2] = dIgdFxA[0] = dIgdFxA[2] = 0.0;
    dgdFBA[0] = dgdFBA[2] = dIgdFBA[0] = dIgdFBA[2] = 0.0;
  }
}

/* Compute derivatives of physical basis functions at magnetic axis */
void FLT_NAME(bf3p_dfA)(void *params, FLT dgAdFBA[3], FLT dIgAdFBA[3], FLT FBA)
{
  meq_bool circ;
  
  circ = ((bf3p_params *) params)->circ;
  
   dgAdFBA[0] = -1.0;             /* -1              */
  dIgAdFBA[0] =  FBA;             /* FB - FA         */
   dgAdFBA[1] =  dgAdFBA[0];      /* -1              */
  dIgAdFBA[1] = dIgAdFBA[0];      /* FB - FA         */
   dgAdFBA[2] =  0.0;             /* 0               */
  dIgAdFBA[2] =  0.5 * FBA * FBA; /* 0.5*(FB - FA)^2 */

  if (circ) {
    /* For low elongation geometry */
    dgAdFBA[0] = dgAdFBA[2] = dIgAdFBA[0] = dIgAdFBA[2] = 0.0;
  }
}

/* Derivatives of conversion factors from normalized to physical basis functions */
void FLT_NAME(bf3p_dfag)(void *params, FLT dalphapg[4], FLT dalphag[4], FLT FBA)
{
  dalphapg[0] =                 1.0; /* 1           */
  dalphapg[1] = dalphapg[0]        ; /* 1           */
  dalphapg[2] =             2.0*FBA; /* 2 (FB-FA)   */
   dalphag[0] = dalphapg[2]        ; /* 2 (FB-FA)   */
   dalphag[1] = dalphapg[2]        ; /* 2 (FB-FA)   */
   dalphag[2] = dalphapg[2]*1.5*FBA; /* 3 (FB-FA)^2 */
}

void FLT_NAME(bf3p_fPg)(FLT fPg[3], FLT fTg[3])
{
  fPg[0] = fTg[1] = fTg[2] = 1.0;
  fTg[0] = fPg[1] = fPg[2] = 0.0;
}

/*
 * These functions are the ones used in Simulink S-function blocks for each
 * mode and in MEX-files when a custom implementation is present
 *  - modes 1,2,3,5,6,8 have a custom implementation
 *  - mode 4 (vertical chord integrals) is not implemented
 *  - mode 7 (inequality constraints) is not implemented
 */

void FLT_NAME(bf3p1)(FLT *Tyg, FLT *Tpg, FLT *ITpg,
                     FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                     FLT *ry, FLT *iry, int nzy, int nry,
                     meq_bool circ, int err) {
 int i, j, ny=nzy*nry;
 FLT *t0=Tyg, *t1=t0+ny, *t2=t1+ny, vF, v0, v1, v2;
 Fx += nzy + 3;
 Tpg[0] = Tpg[1] = Tpg[2] = ITpg[0] = ITpg[1] = ITpg[2] = FLTC(0.0);
 FA -= FB;
 if (err) for (i=ny; i--;) *t0++ = *t1++ = *t2++ = FLTC(0.0);
 else if (circ) {
  for (i=nry; i--; iry++, Fx+=2)
   for (j=nzy; j--; Fx++) {
    *t0++ = *t2++ = FLTC(0.0);
    if (*Opy++) { Tpg[1] += (v1 = (vF =*Fx - FB) * *iry); *t1++ = v1; ITpg[1] += FLTC(0.5) * vF * v1; }
    else         *t1++ = FLTC(0.0);
   }
 } else
  for (i=nry; i--; ry++, iry++, Fx+=2)
   for (j=nzy; j--; Fx++)
    if (*Opy++) {
     Tpg[0] += (v0 = (vF = *Fx - FB) * *ry); *t0++ = v0;
     Tpg[1] += (v1 = vF * *iry)            ; *t1++ = v1; 
     Tpg[2] += (v2 = (vF - FA) * v1)       ; *t2++ = v2;
     ITpg[0] += ((v1 = FLTC(0.5) * vF * vF) * *ry);
     ITpg[1] += (v1 *= *iry);
     ITpg[2] += (FLTC(0.66666666666666666666) * vF - FA) * v1;
    } else *t0++ = *t1++ = *t2++ = FLTC(0.0);
}

void FLT_NAME(bf3p2)(FLT *gQg, FLT *IgQg, FLT *FNQ, int nQ) {
 int i;
 FLT *t0=gQg, *t1=t0+nQ, *t2=t1+nQ, *T0=IgQg, *T1=T0+nQ, *T2=T1+nQ, vF, v0;
 for (i=nQ; i--;) {
  *t0++ = *t1++ = v0 = (vF = *FNQ++) - FLTC(1.0);
  *t2++ = v0 * vF;
  *T0++ = *T1++ = (v0 = FLTC(0.5) * v0 * v0);
  *T2++ = ( vF + FLTC(0.5) ) * FLTC(0.66666666666666666666) * v0;
 }
}

void FLT_NAME(bf3p3)(FLT *aPpg, FLT *aTTpg, FLT *aPg, FLT *ahqTg, FLT *ag, FLT FA, FLT FB, FLT ids) {
 FLT cP=ids*FLTC(0.159154943091895 /* 1/2/pi */), cT=ids*FLTC(2e-7 /* mu0/2/pi */);
 FB -= FA;
   aPg[0] = (  aPpg[0] = ag[0] * FB *      cP) * FB;
 ahqTg[1] = ( aTTpg[1] = ag[1] * FB *      cT) * FB;
 ahqTg[2] = ( aTTpg[2] = ag[2] * FB * FB * cT) * FB;
  aPpg[1] = aPpg[2] = aTTpg[0] = aPg[1] = aPg[2] = ahqTg[0] = FLTC(0.0);  
}

void FLT_NAME(bf3p5)(FLT *gA, FLT *IgA, FLT FA, FLT FB, meq_bool circ) {
 FLT FAB = FA-FB, FAB2=FLTC(0.5)*FAB*FAB;
 gA[1] = FAB; IgA[1] = FAB2; gA[2] = FLTC(0.0);
 if (circ) {
  gA[0] =      IgA[0] = IgA[2] = FLTC(0.0);
 } else {
  gA[0] = FAB; IgA[0] = FAB2;
               IgA[2] = FLTC(-0.333333333333333333)*FAB2*FAB;
 }
}

void FLT_NAME(bf3p6)(FLT *Qg, FLT *Xq, FLT FA, FLT FB, FLT irA, FLT ids) {
 FB -= FA;
 Qg[0] = Qg[1] = Xq[0] = FLTC(0.0);
 Qg[2] = FLTC(2.0)*FB*FB*irA*ids;
}

void FLT_NAME(bf3p8)(FLT *Bty, FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                     FLT *ag, FLT rBt, int err, FLT ids, FLT *iry, int nzy, int nry) {
 int i, j;
 FLT v1, a1, a2;
 ids *= FLTC(1e-7 /* mu0/4/pi */)/rBt;
 FA -= FB;
 a2  = ag[2]*ids;
 a1  = ag[1]*ids - FA*a2;
 a2 *= FLTC(0.66666666666666666666);
 Fx += nzy + 3;
 if (err) for (i=nzy*nry; i--;) *Bty++ = FLTC(0.0);
 else
  for (i=nry; i--; iry++, Fx+=2)
   for (j=nzy; j--; Fx++)
    if (*Opy++) {
     v1 = *Fx - FB;
     *Bty++ = ((a2*v1 + a1)*v1*v1 + rBt) * *iry;
    } else *Bty++ = rBt * *iry;
}

void FLT_NAME(bf3p16)(FLT *dQqgdFA, FLT *dQqgdFB, FLT *dQqgdrA, FLT *dQqgdirA,
                      FLT FA, FLT FB, FLT irA, FLT ids) {
 FB -= FA;
 dQqgdFA[0] = dQqgdFB[0] = dQqgdrA[0] = dQqgdirA[0] = FLTC(0.0);
 dQqgdFA[1] = dQqgdFB[1] = dQqgdrA[1] = dQqgdirA[0] = FLTC(0.0);
 dQqgdFA[2] = -FLTC(4.0)*FB*irA*ids;
 dQqgdFB[2] =  FLTC(4.0)*FB*irA*ids;
 dQqgdrA[2] =  FLTC(0.0);
 dQqgdirA[2] = FLTC(2.0)*FB*FB *ids;
}

